<?php
require_once 'db.php';
require_once 'schema_guard.php';
require_once 'auth.php';
require_once 'ui.php';

require_login();
sb_ensure_attendance_schema($conn);

@date_default_timezone_set('Asia/Karachi');

$uid   = current_user_id();
$admin = is_admin();

function sb_parse_time_to_hms(string $t): string {
  $t = trim($t);
  if ($t === '') return '';
  // allow HH:MM or HH:MM:SS
  if (preg_match('/^\d{2}:\d{2}$/', $t)) return $t . ':00';
  if (preg_match('/^\d{2}:\d{2}:\d{2}$/', $t)) return $t;
  return '';
}
function sb_fmt_time_ampm(?string $t): string {
  $t = trim((string)$t);
  if ($t === '' || $t === '00:00:00') return '-';
  $ts = strtotime('1970-01-01 ' . $t);
  if ($ts === false) return $t;
  return date('h:i A', $ts);
}

$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
if ($id <= 0) { header("Location: attendance_list.php?e=missing_id"); exit(); }

$st = mysqli_prepare($conn, "SELECT a.*, s.name AS staff_name, s.id AS staff_id
                             FROM attendance a
                             JOIN staff_account s ON s.id=a.staff_id
                             WHERE a.id=? LIMIT 1");
if (!$st) { die("DB error"); }
mysqli_stmt_bind_param($st, "i", $id);
mysqli_stmt_execute($st);

// mysqlnd-free fetch one row
$meta = mysqli_stmt_result_metadata($st);
if (!$meta) { mysqli_stmt_close($st); header("Location: attendance_list.php?e=not_found"); exit(); }
$fields = mysqli_fetch_fields($meta);
$row = [];
$bind = [];
foreach ($fields as $f) { $row[$f->name] = null; $bind[] = &$row[$f->name]; }
mysqli_stmt_bind_result($st, ...$bind);
$data = null;
if (mysqli_stmt_fetch($st)) {
  $data = [];
  foreach ($row as $k=>$v) $data[$k]=$v;
}
mysqli_free_result($meta);
mysqli_stmt_close($st);
if (!$data) { header("Location: attendance_list.php?e=not_found"); exit(); }

// Staff can edit only own attendance
if (!$admin) {
  $my_staff_id = 0;
  $q = mysqli_prepare($conn, "SELECT id FROM staff_account WHERE user_id=? LIMIT 1");
  if ($q) {
    mysqli_stmt_bind_param($q, "i", $uid);
    mysqli_stmt_execute($q);
    mysqli_stmt_bind_result($q, $my_staff_id);
    mysqli_stmt_fetch($q);
    mysqli_stmt_close($q);
  }
  if ((int)$my_staff_id !== (int)$data['staff_id']) {
    header("Location: attendance_list.php?e=not_allowed");
    exit();
  }
}


$msg = '';
$err = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!csrf_validate($_POST['csrf'] ?? '')) {
    $err = "Invalid session token.";
  } else {
    $in_time       = sb_parse_time_to_hms((string)($_POST['in_time'] ?? ''));
    $out_time      = sb_parse_time_to_hms((string)($_POST['out_time'] ?? ''));
    $home_in_time  = sb_parse_time_to_hms((string)($_POST['home_in_time'] ?? ''));
    $home_out_time = sb_parse_time_to_hms((string)($_POST['home_out_time'] ?? ''));

    if ($in_time === '') { $err = "IN time is required."; }

    // compute total hours (office) if out provided; handle overnight by allowing out < in (next day)
    $total_hours = null;
    $overtime_hours = null;

    if ($err === '' && $out_time !== '') {
      $in_ts  = strtotime('1970-01-01 ' . $in_time);
      $out_ts = strtotime('1970-01-01 ' . $out_time);
      if ($in_ts !== false && $out_ts !== false) {
        if ($out_ts < $in_ts) $out_ts += 86400; // next day
        $hrs = round(($out_ts - $in_ts) / 3600, 2);
        if ($hrs < 0) $hrs = 0;
        $total_hours = number_format($hrs, 2, '.', '');
        // OT: above 8 hours (simple rule)
        $ot = $hrs - 8;
        if ($ot < 0) $ot = 0;
        $overtime_hours = number_format(round($ot, 2), 2, '.', '');
      }
    }

    if ($err === '') {
      $sql = "UPDATE attendance
              SET in_time=?,
                  out_time=?,
                  home_in_time=?,
                  home_out_time=?,
                  total_hours=COALESCE(?, total_hours),
                  overtime_hours=COALESCE(?, overtime_hours),
                  updated_by=?,
                  updated_at=NOW()
              WHERE id=? LIMIT 1";
      $u = mysqli_prepare($conn, $sql);
      if (!$u) {
        $err = "DB prepare failed.";
      } else {
        // Use nulls safely
        $out_b = ($out_time === '') ? null : $out_time;
        $hin_b = ($home_in_time === '') ? null : $home_in_time;
        $hout_b= ($home_out_time === '') ? null : $home_out_time;
        mysqli_stmt_bind_param($u, "ssssssii",
          $in_time,
          $out_b,
          $hin_b,
          $hout_b,
          $total_hours,
          $overtime_hours,
          $uid,
          $id
        );
        if (mysqli_stmt_execute($u)) {
          mysqli_stmt_close($u);
          header("Location: attendance_list.php?m=updated");
          exit();
        } else {
          $err = "Update failed.";
          mysqli_stmt_close($u);
        }
      }
    }
  }
}

// Defaults for form (HH:MM)
function sb_hm($t){
  $t = trim((string)$t);
  if ($t === '' || $t === '00:00:00') return '';
  return substr($t,0,5);
}
?>
<!doctype html>
<html>
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Edit Attendance</title>
<style>
body{font-family:system-ui,Arial;margin:0;background:#eef2f7}
.card{max-width:720px;margin:18px auto;background:#fff;border-radius:18px;box-shadow:0 18px 40px rgba(2,6,23,.12);padding:18px}
h2{margin:0 0 10px}
.meta{color:#475569;font-weight:700;margin-bottom:14px}
.row{display:flex;gap:12px;flex-wrap:wrap}
.field{flex:1;min-width:220px}
label{display:block;font-weight:900;color:#334155;font-size:13px;margin-bottom:6px}
input[type=time]{width:100%;padding:12px;border:1px solid rgba(15,23,42,.15);border-radius:14px;font-size:15px}
.btns{display:flex;gap:10px;flex-wrap:wrap;margin-top:14px}
button,a.btn{padding:11px 16px;border-radius:14px;border:0;font-weight:1000;cursor:pointer;text-decoration:none;display:inline-block}
button{background:#2563eb;color:#fff}
a.btn{background:#e2e8f0;color:#0f172a}
.alert{padding:10px 12px;border-radius:14px;margin:10px 0;font-weight:900}
.ok{background:#e8f5e9;color:#1b5e20}
.err{background:#ffebee;color:#b71c1c}
.small{font-size:12px;color:#64748b;font-weight:800;margin-top:6px}
</style>

<link rel="stylesheet" href="assets/sb_mobile.css?v=20260128">
<script src="assets/sb_mobile.js?v=20260128" defer></script>
</head>
<body>
<div class="card">
  <h2>✏️ Edit Attendance</h2>
  <div class="meta">
    Date: <b><?php echo htmlspecialchars((string)$data['attendance_date']); ?></b> —
    Staff: <b><?php echo htmlspecialchars((string)$data['staff_name']); ?></b><br>
    Current: IN <?php echo htmlspecialchars(sb_fmt_time_ampm((string)$data['in_time'])); ?>,
    OUT <?php echo htmlspecialchars(sb_fmt_time_ampm((string)$data['out_time'])); ?>
  </div>

  <?php if($msg): ?><div class="alert ok"><?php echo htmlspecialchars($msg); ?></div><?php endif; ?>
  <?php if($err): ?><div class="alert err"><?php echo htmlspecialchars($err); ?></div><?php endif; ?>

  <form method="post">
    <input type="hidden" name="csrf" value="<?php echo htmlspecialchars(csrf_token()); ?>">

    <div class="row">
      <div class="field">
        <label>Office IN</label>
        <input type="time" name="in_time" value="<?php echo htmlspecialchars(sb_hm($data['in_time'])); ?>" required>
      </div>
      <div class="field">
        <label>Office OUT</label>
        <input type="time" name="out_time" value="<?php echo htmlspecialchars(sb_hm($data['out_time'])); ?>">
        <div class="small">Optional. If OUT is next day, system will handle automatically.</div>
      </div>
    </div>

    <div class="row" style="margin-top:10px">
      <div class="field">
        <label>Home IN</label>
        <input type="time" name="home_in_time" value="<?php echo htmlspecialchars(sb_hm($data['home_in_time'] ?? '')); ?>">
      </div>
      <div class="field">
        <label>Home OUT</label>
        <input type="time" name="home_out_time" value="<?php echo htmlspecialchars(sb_hm($data['home_out_time'] ?? '')); ?>">
      </div>
    </div>

    <div class="btns">
      <button type="submit">💾 Save</button>
      <a class="btn" href="attendance_list.php">⬅ Back to List</a>
      <a class="btn" href="dashboard.php">🏠 Dashboard</a>
    </div>
  </form>
</div>
</body>
</html>
