<?php
require_once 'db.php';
require_once 'schema_guard.php';
require_once 'auth.php';
require_once 'ui.php';

require_login();
require_permission($conn, 'attendance_in');

sb_ensure_attendance_schema($conn);

@date_default_timezone_set('Asia/Karachi');

 /**
  * =========================
  * ATTENDANCE IN SETTINGS
  * =========================
  * If ENFORCE_WINDOW is false, staff can mark IN anytime.
  */
 $ATTENDANCE_IN_ENFORCE_WINDOW = false; // ✅ open for all staff, anytime
 $ATTENDANCE_IN_START = '08:00:00';     // used only if ENFORCE_WINDOW = true
 $ATTENDANCE_IN_END   = '12:00:00';     // used only if ENFORCE_WINDOW = true



$uid  = current_user_id();
$role = current_role();

$today    = date('Y-m-d');
$now_time = date('H:i:s');
$now_time_ampm = date('h:i A');

$msg  = '';
$type = 'warn';

// Staff link
$staff_id = 0;
$st = mysqli_prepare($conn, "SELECT id, name FROM staff_account WHERE user_id=? LIMIT 1");
$staff_name = '';
if ($st) {
  mysqli_stmt_bind_param($st, "i", $uid);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st, $sid, $sname);
  if (mysqli_stmt_fetch($st)) {
    $staff_id = (int)$sid;
    $staff_name = (string)$sname;
  }
  mysqli_stmt_close($st);
}


// Admin can choose any staff if not linked
$admin_pick_staff = false;
if ($staff_id <= 0 && is_admin()) {
  $admin_pick_staff = true;
  $picked = (int)($_POST['staff_id'] ?? ($_SESSION['pick_staff_id'] ?? 0));
  if ($picked > 0) {
    $ps = mysqli_prepare($conn, "SELECT id, name FROM staff_account WHERE id=? LIMIT 1");
    if ($ps) {
      mysqli_stmt_bind_param($ps, "i", $picked);
      mysqli_stmt_execute($ps);
      mysqli_stmt_bind_result($ps, $pid, $pname);
      if (mysqli_stmt_fetch($ps)) {
        $staff_id = (int)$pid;
        $staff_name = (string)$pname;
        $_SESSION['pick_staff_id'] = $staff_id;
      }
      mysqli_stmt_close($ps);
    }
  }
}

if ($staff_id <= 0 && !is_admin()) {
  sb_page_header('Attendance IN');
  echo '<div class="msg err">Your user is not linked with staff account. Admin please set staff_account.user_id = users.id.</div>';
  sb_page_footer();
  exit;
}

$csrf = csrf_token();

// Default time shown in form (editable)
$form_time = $now_time;

if (isset($_POST['checkin'])) {
  if (!csrf_validate($_POST['csrf'] ?? '')) {
    $msg = 'Security error (CSRF). Please refresh.';
    $type = 'err';
  } else {
    // Editable time (HH:MM or HH:MM:SS)
    $raw_time = trim((string)($_POST['in_time'] ?? ''));
    if ($raw_time !== '') {
      if (!preg_match('/^\\d{2}:\\d{2}(:\\d{2})?$/', $raw_time)) {
        $msg = 'Invalid time format.';
        $type = 'err';
      } else {
        // Normalize to HH:MM:SS
        $form_time = (strlen($raw_time) === 5) ? ($raw_time . ':00') : $raw_time;
      }
    }


    if ($form_time === '') { $form_time = date('H:i:s'); }

if ($type === 'err') {
      // stop further processing
    } else {
    // Attendance window (staff only) - configurable
    $window_ok = true;
    if ($ATTENDANCE_IN_ENFORCE_WINDOW && !is_admin()) {
      $start_ts  = strtotime($today . ' ' . $ATTENDANCE_IN_START);
      $end_ts    = strtotime($today . ' ' . $ATTENDANCE_IN_END);
      $now_ts    = time();

      if ($start_ts !== false && $end_ts !== false) {
        if ($now_ts < $start_ts) {
          $window_ok = false;
          $msg = 'Attendance IN allowed from ' . date('h:i A', $start_ts) . '.';
          $type = 'warn';
        } elseif ($now_ts > $end_ts) {
          $window_ok = false;
          $msg = 'Attendance IN allowed only till ' . date('h:i A', $end_ts) . '.';
          $type = 'warn';
        }
      }
    }

    if (!$window_ok) {
      // stop
    } else {
      // Check existing today
      $existing = null;
      $q = mysqli_prepare($conn, "SELECT id, in_time FROM attendance WHERE staff_id=? AND attendance_date=? LIMIT 1");
      if ($q) {
        mysqli_stmt_bind_param($q, "is", $staff_id, $today);
        mysqli_stmt_execute($q);
        // mysqlnd-free fetch
        mysqli_stmt_bind_result($q, $ex_id, $ex_in_time);
        $existing = null;
        if (mysqli_stmt_fetch($q)) {
          $existing = ['id' => (int)$ex_id, 'in_time' => (string)$ex_in_time];
        }
        mysqli_stmt_close($q);
      }

      if ($existing && !empty($existing['in_time'])) {
        $msg = 'You already checked IN today.';
        $type = 'warn';
      } elseif ($existing && empty($existing['in_time'])) {
        // Update missing IN
        $u = mysqli_prepare($conn, "UPDATE attendance SET in_time=?, user_id=?, created_by=COALESCE(created_by,?), updated_by=?, updated_at=NOW() WHERE id=? LIMIT 1");
        if ($u) {
          $eid = (int)$existing['id'];
          mysqli_stmt_bind_param($u, "siiii", $form_time, $uid, $uid, $uid, $eid);
          if (mysqli_stmt_execute($u)) {
            $msg = '✅ Attendance IN saved.';
            $type = 'ok';
          } else {
            $msg = '❌ Update failed: ' . htmlspecialchars(mysqli_stmt_error($u));
            $type = 'err';
          }
          mysqli_stmt_close($u);
        } else {
          $msg = 'Server error (prepare failed).';
          $type = 'err';
        }
      } else {
        // Insert
        $ins = mysqli_prepare($conn, "INSERT INTO attendance (staff_id, attendance_date, in_time, user_id, created_by, updated_by, created_at, updated_at) VALUES (?,?,?,?,?,?,NOW(),NOW())");
        if ($ins) {
          mysqli_stmt_bind_param($ins, "issiii", $staff_id, $today, $form_time, $uid, $uid, $uid);
          if (mysqli_stmt_execute($ins)) {
            $msg = '✅ Attendance IN saved.';
            $type = 'ok';
          } else {
            $msg = '❌ Insert failed: ' . htmlspecialchars(mysqli_stmt_error($ins));
            $type = 'err';
          }
          mysqli_stmt_close($ins);
        } else {
          $msg = 'Server error (prepare failed).';
          $type = 'err';
        }
      }
    }
    }
  }
}

sb_page_header('Attendance IN');

echo '<div class="muted">Date: <b>' . htmlspecialchars($today) . '</b> &nbsp; Time: <b>' . htmlspecialchars($form_time) . '</b></div>';
if ($staff_name !== '') {
  echo '<div class="muted" style="margin-top:6px">Staff: <b>' . htmlspecialchars($staff_name) . '</b></div>';
}

if ($msg !== '') {
  echo '<div class="msg ' . htmlspecialchars($type) . '">' . htmlspecialchars($msg) . '</div>';
}
?>

<style>
:root{--ink:#0f172a;--muted:#475569;--stroke:rgba(15,23,42,.10);--shadow:0 18px 44px rgba(2,6,23,.16);--r:22px;}
body{
  margin:0;
  font-family:system-ui,Arial;
  color:var(--ink);
  background-color:#e9eef6;
  background-image:
    radial-gradient(1200px 520px at 15% 10%, rgba(37,99,235,.22), transparent 60%),
    radial-gradient(900px 420px at 85% 15%, rgba(71,85,105,.30), transparent 60%),
    radial-gradient(1100px 520px at 55% 95%, rgba(59,130,246,.16), transparent 65%),
    radial-gradient(760px 420px at 10% 90%, rgba(148,163,184,.40), transparent 60%);
}
/* hide reset links (header) */
a[href*="forgot_password.php"], a[href*="reset"]{display:none!important}

.sb-card{
  max-width: 980px;
  margin: 18px auto 22px;
  padding: 18px;
  border-radius: var(--r);
  background: rgba(255,255,255,.78);
  backdrop-filter: blur(10px);
  border:1px solid rgba(255,255,255,.45);
  box-shadow: var(--shadow);
}
.sb-title{font-size:24px;font-weight:1100;letter-spacing:.2px;margin:0 0 10px 0}
.sb-meta{
  display:flex;gap:10px;flex-wrap:wrap;align-items:center;
  margin:8px 0 14px 0;
}
.pill{
  padding:10px 14px;border-radius:999px;
  background:rgba(255,255,255,.72);
  border:1px solid var(--stroke);
  box-shadow: 0 10px 20px rgba(2,6,23,.08);
  font-weight:1000;
}
.pill small{color:var(--muted);font-weight:900}

.sb-panel{
  position:relative;
  border-radius: 20px;
  padding: 18px;
  background: rgba(255,255,255,.70);
  border:1px solid rgba(15,23,42,.10);
  overflow:hidden;
}
.sb-panel:before{
  content:"";
  position:absolute;inset:-2px;
  background-image:
    radial-gradient(circle at 1px 1px, rgba(15,23,42,.08) 1px, transparent 1.9px),
    radial-gradient(720px 260px at 12% 0%, rgba(37,99,235,.22), transparent 60%),
    radial-gradient(680px 260px at 90% 10%, rgba(71,85,105,.20), transparent 60%),
    radial-gradient(650px 280px at 40% 100%, rgba(59,130,246,.14), transparent 60%);
  background-size: 18px 18px, auto, auto, auto;
  opacity:.90;
  pointer-events:none;
}
.sb-panel > *{position:relative;z-index:1}

.sb-row{display:flex;gap:14px;flex-wrap:wrap;align-items:end;}
.sb-field{flex:1;min-width:260px;}
.sb-field label{display:block;font-size:13px;color:var(--muted);margin:0 0 6px 2px;font-weight:1000;}
.sb-field input[type=time], .sb-field input[type=text], .sb-field select, textarea{
  width:100%;
  padding:14px 16px;
  border:1px solid rgba(120,140,160,.30);
  border-radius:16px;
  font-size:16px;
  outline:none;
  background:rgba(255,255,255,.88);
  box-shadow: 0 8px 16px rgba(2,6,23,.06) inset;
}
textarea{min-height:120px;resize:vertical}
.sb-actions,.sb-cta{display:flex;gap:12px;flex-wrap:wrap;margin-top:16px}
.btn.big{
  padding:14px 18px;border-radius:18px;font-size:16px;font-weight:1100;letter-spacing:.2px;border:0;cursor:pointer;
  box-shadow:0 14px 28px rgba(2,6,23,.12);
}
.btn.primary.big{background:linear-gradient(135deg, rgba(37,99,235,.95), rgba(71,85,105,.70));color:#fff}
.btn.danger.big{background:linear-gradient(135deg, rgba(239,68,68,.92), rgba(71,85,105,.70));color:#fff}
.btn.out.big{background:linear-gradient(135deg, rgba(71,85,105,.92), rgba(37,99,235,.55));color:#fff;text-decoration:none;display:inline-flex;align-items:center;justify-content:center}
.sb-note{font-size:12px;color:var(--muted);margin-top:10px;font-weight:800}

.msg{max-width:980px;margin:10px auto 0;padding:0 12px}
.msg .box{padding:12px 14px;border-radius:18px;background:rgba(255,255,255,.86);border:1px solid var(--stroke)}
.msg .ok{border-color:rgba(34,197,94,.45)}
.msg .warn{border-color:rgba(250,204,21,.55)}
.msg .err{border-color:rgba(239,68,68,.45);color:#8a1f1f}
</style>

<div class="sb-card">
  <div class="sb-title">Attendance IN</div>
<div class="sb-meta">
  <div class="pill"><small>Date:</small> <?php echo htmlspecialchars($today); ?></div>
  <div class="pill"><small>Time:</small> <?php echo htmlspecialchars($now_time_ampm); ?></div>
  <div class="pill"><small>Staff:</small> <?php echo htmlspecialchars($staff_name); ?></div>
</div>


<div class="sb-panel">
<form method="post" autocomplete="off">
  <input type="hidden" name="csrf" value="<?php echo htmlspecialchars($csrf); ?>">
<?php if (!empty($admin_pick_staff)) { ?>
  <div class="sb-row" style="margin-top:10px">
    <div class="sb-field">
      <label>Select Staff (Admin)</label>
      <select name="staff_id" required style="width:100%;padding:12px 14px;border:1px solid rgba(120,140,160,.35);border-radius:14px;font-size:15px;background:#fff;">
        <option value="">-- Select Staff --</option>
        <?php
          $rs = mysqli_query($conn, "SELECT id, name FROM staff_account ORDER BY name");
          $cur = (int)$staff_id;
          if ($rs) {
            while ($r = mysqli_fetch_assoc($rs)) {
              $id = (int)$r['id'];
              $nm = (string)$r['name'];
              $sel = ($id === $cur) ? 'selected' : '';
              echo '<option value="'.htmlspecialchars((string)$id).'" '.$sel.'>'.htmlspecialchars($nm).' (ID '.$id.')</option>';
            }
            mysqli_free_result($rs);
          }
        ?>
      </select>
      <div class="sb-note">Admin: pehle staff select karein, phir IN mark karein.</div>
    </div>
  </div>
<?php } ?>

  <div class="sb-row">
    <div class="sb-field">
      <label>Check-in Time (editable)</label>
      <input type="time" name="in_time" step="1" value="<?php echo htmlspecialchars(substr($form_time,0,8)); ?>">
      <div class="sb-note">Tip: Default current time. You can edit if needed.</div>
    </div>
  </div>

  <div class="sb-actions">
    <button class="btn primary big" type="submit" name="checkin">🟢 CHECK IN</button>
    <a class="btn out big" href="attendance_out.php">➡️ GO TO OUT</a>
  </div>
</form>
</div>

<?php sb_page_footer();