<?php
$today = date('Y-m-d');
$from_date = $today;
$to_date = $today;
require_once 'sb_session.php';
require_once 'db.php';
require_once 'delete_log.php';

@date_default_timezone_set('Asia/Karachi');

if (empty($_SESSION['admin_id']) && empty($_SESSION['user_id'])) {
    header("Location: login.php");
    exit();
}

$is_admin = (!empty($_SESSION['admin']) || !empty($_SESSION['admin_id']));

$actor_id = (int)($_SESSION['admin_id'] ?? $_SESSION['user_id'] ?? 0);
$msg = '';
$err = '';

// Date filter (GET)
// IMPORTANT: If user did not apply any filter, show ALL records (no date WHERE clause)
// so the list doesn't look empty when there is no attendance for "today".
$from = (string)($_GET['from'] ?? '');
$to   = (string)($_GET['to'] ?? '');
$from_ok = ($from === '') ? true : (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $from);
$to_ok   = ($to === '') ? true : (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $to);
if (!$from_ok) { $from = ''; }
if (!$to_ok)   { $to   = ''; }

// Helper: fetch rows from prepared statement (mysqlnd-free)
function sb_stmt_fetch_all_assoc(mysqli_stmt $st): array {
    // mysqlnd-free fetch: DO NOT call mysqli_stmt_get_result() unless it exists
    // (many shared hosting builds don't have mysqlnd, causing fatal error).
    $rows = [];
    if (function_exists('mysqli_stmt_get_result')) {
        $res = @mysqli_stmt_get_result($st);
        if ($res) {
            while ($r = mysqli_fetch_assoc($res)) { $rows[] = $r; }
            mysqli_free_result($res);
            return $rows;
        }
    }

    $meta = mysqli_stmt_result_metadata($st);
    if (!$meta) return $rows;

    $fields = mysqli_fetch_fields($meta);
    $row = [];
    $bind = [];
    foreach ($fields as $f) { $row[$f->name] = null; $bind[] = &$row[$f->name]; }
    mysqli_stmt_bind_result($st, ...$bind);

    while (mysqli_stmt_fetch($st)) {
        $copy = [];
        foreach ($row as $k => $v) { $copy[$k] = $v; }
        $rows[] = $copy;
    }
    mysqli_free_result($meta);
    return $rows;
}

function sb_fmt_ampm($timeStr): string {
    $timeStr = trim((string)$timeStr);
    if ($timeStr === '') return '-';
    $ts = strtotime($timeStr);
    if ($ts === false) return $timeStr;
    return date('h:i A', $ts);
}
function sb_home_status(array $r): string {
    $hin  = trim((string)($r['home_in_time'] ?? ''));
    $hout = trim((string)($r['home_out_time'] ?? ''));
    if ($hin !== '' && $hout === '') return 'HOME (IN)';
    if ($hin !== '' && $hout !== '') return 'HOME (OUT)';
    return '-';
}


/* Handle delete from list (with audit log) */
if ($is_admin) {
if (isset($_GET['delete'])) {
    $id = (int)$_GET['delete'];
    if ($id > 0) {
        // Snapshot
        $snap = null;
        $st = mysqli_prepare($conn, "SELECT * FROM attendance WHERE id=? LIMIT 1");
        if ($st) {
            mysqli_stmt_bind_param($st, "i", $id);
            mysqli_stmt_execute($st);
            $rows = sb_stmt_fetch_all_assoc($st);
            $snap = $rows[0] ?? null;
            mysqli_stmt_close($st);
        }

        $ok = false;
        if ($snap) {
            // Log delete
            @sb_log_deleted_record($conn, 'attendance', $id, $actor_id, 'Deleted from attendance list', $snap);
        }

        $del = mysqli_prepare($conn, "DELETE FROM attendance WHERE id=? LIMIT 1");
        if ($del) {
            mysqli_stmt_bind_param($del, "i", $id);
            $ok = (bool)mysqli_stmt_execute($del);
            mysqli_stmt_close($del);
        }

        if ($ok) {
            $msg = "✅ Record deleted.";
        } else {
            $err = "❌ Delete failed.";
        }
    }
}
}

?>
<!DOCTYPE html>
<html>
<head>
<title>Attendance List</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">

<style>
:root{
  --ink:#0f172a;
  --muted:#475569;
  --stroke:rgba(15,23,42,.10);
  --shadow: 0 18px 40px rgba(2,6,23,.14);
  --r:22px;
}
body{
  margin:0;
  font-family:system-ui,Arial;
  color:var(--ink);
  /* Watercolor blue + grey background */
  background-color:#eef2f7;
  background-image:
    radial-gradient(1200px 520px at 15% 10%, rgba(59,130,246,.22), transparent 60%),
    radial-gradient(900px 420px at 85% 15%, rgba(148,163,184,.32), transparent 60%),
    radial-gradient(1100px 520px at 55% 95%, rgba(37,99,235,.18), transparent 65%),
    radial-gradient(760px 420px at 10% 90%, rgba(203,213,225,.45), transparent 60%);
}

/* 1-inch animated strip */
.topstrip{
  height:96px;
  display:flex;
  align-items:center;
  justify-content:center;
  position:sticky;
  top:0;
  z-index:999;
  border-bottom:1px solid rgba(255,255,255,.22);
  background:
    linear-gradient(90deg,
      rgba(2,6,23,.92),
      rgba(29,78,216,.35),
      rgba(22,163,74,.32),
      rgba(220,38,38,.28),
      rgba(2,6,23,.92)
    );
  background-size: 240% 100%;
  animation: stripMove 7s ease-in-out infinite alternate;
}
@keyframes stripMove{
  0%{background-position:0% 50%; filter:brightness(.92);}
  50%{background-position:60% 50%; filter:brightness(1.05);}
  100%{background-position:100% 50%; filter:brightness(.95);}
}
.topstrip .inner{
  width:min(1400px, 100%);
  padding:0 14px;
  display:flex;
  align-items:center;
  justify-content:space-between;
  gap:14px;
}
.brand{
  color:#fff;
  font-weight:1100;
  letter-spacing:.4px;
  font-size:22px;
  display:flex;
  align-items:center;
  gap:10px;
  text-transform:uppercase;
}
.navlinks{
  display:flex;
  gap:10px;
  flex-wrap:wrap;
  justify-content:flex-end;
}
.navlinks a{
  color:#0f172a;
  background:rgba(255,255,255,.78);
  border:1px solid rgba(255,255,255,.25);
  text-decoration:none;
  padding:10px 14px;
  border-radius:16px;
  font-weight:1000;
  box-shadow: 0 10px 20px rgba(2,6,23,.18);
  backdrop-filter: blur(10px);
}
.navlinks a:hover{filter:brightness(1.03)}

/* Container / list sizing */
.container{
  max-width:1400px;
  margin:16px auto 22px;
  background:rgba(255,255,255,.78);
  backdrop-filter: blur(10px);
  padding:18px;
  border-radius:var(--r);
  box-shadow: var(--shadow);
  border:1px solid rgba(255,255,255,.45);
  overflow:auto;
}
h2{margin:6px 0 12px;font-size:22px;letter-spacing:.2px}

.filter{
  display:flex;
  gap:12px;
  flex-wrap:wrap;
  align-items:end;
  margin:10px 0 16px;
}
.filter label{
  font-size:12.5px;
  color:var(--muted);
  display:flex;
  flex-direction:column;
  gap:6px;
  font-weight:900;
}
.filter input{
  padding:10px 12px;
  border:1px solid rgba(120,140,160,.30);
  border-radius:14px;
  min-width:190px;
  font-size:14px;
  background:rgba(255,255,255,.86);
}
.filter button{
  padding:11px 16px;
  border:0;
  border-radius:14px;
  background:linear-gradient(135deg, rgba(59,130,246,.92), rgba(148,163,184,.70));
  color:#fff;
  cursor:pointer;
  font-weight:1000;
  box-shadow: 0 12px 22px rgba(59,130,246,.20);
}
.filter a{
  padding:11px 16px;
  border-radius:14px;
  background:rgba(255,255,255,.70);
  color:var(--ink);
  text-decoration:none;
  display:inline-block;
  border:1px solid var(--stroke);
  font-weight:1000;
}

table{width:100%;border-collapse:separate;border-spacing:0;min-width:1100px}
th,td{
  padding:12px 10px;
  border-bottom:1px solid rgba(15,23,42,.08);
  text-align:center;
  font-size:14px;
  white-space:nowrap;
}
th{
  position:sticky;
  top:96px; /* below topstrip */
  z-index:5;
  color:#0f172a;
  font-weight:1100;
  background-image:
    linear-gradient(90deg,
      rgba(59,130,246,.20),
      rgba(148,163,184,.26),
      rgba(34,197,94,.16),
      rgba(59,130,246,.20)
    );
  background-size: 240% 100%;
  animation: thWave 8s ease-in-out infinite alternate;
}
@keyframes thWave{0%{background-position:0% 50%}100%{background-position:100% 50%}}
tr:hover td{background:rgba(59,130,246,.06)}
.date-row td{
  position:sticky;
  top:96px; /* keep consistent */
  z-index:4;
  background:rgba(226,232,240,.75) !important;
}

.pill2{
  display:inline-block;
  padding:6px 10px;
  border-radius:999px;
  background:rgba(59,130,246,.12);
  border:1px solid rgba(59,130,246,.18);
  font-weight:1000;
  color:var(--ink);
}
.action a{padding:7px 12px;border-radius:12px;color:#fff;text-decoration:none;font-size:12.5px;font-weight:1000;display:inline-block}
.edit{background:#2e7d32;}
.delete{background:#c62828;}
</style>


<link rel="stylesheet" href="assets/sb_mobile.css?v=20260128">
<script src="assets/sb_mobile.js?v=20260128" defer></script>
</head>

<body>

<div class="topstrip">
  <div class="inner">
    <div class="brand">Study Bridge</div>
    <div class="navlinks">
      <a href="dashboard.php">🏠 Dashboard</a>
      <a href="attendance_list.php">📋 Attendance List</a>
    </div>
  </div>
</div>

<div class="wrap">
<h2>📋 Attendance List</h2>
<?php
if($msg){echo '<div style="padding:10px;margin:10px 0;border-radius:8px;background:#e8f5e9;color:#1b5e20;">'.htmlspecialchars($msg).'</div>'; }
if($err){echo '<div style="padding:10px;margin:10px 0;border-radius:8px;background:#ffebee;color:#b71c1c;">'.htmlspecialchars($err).'</div>'; }
?>

<!-- Date filter -->
<form class="filter" method="get" action="">
    <label>From Date
        <input type="date" name="from" value="<?php echo htmlspecialchars($from); ?>">
    </label>
    <label>To Date
        <input type="date" name="to" value="<?php echo htmlspecialchars($to); ?>">
    </label>
    <button type="submit">Filter</button>
    <a href="attendance_list.php?from=<?php echo rawurlencode($today); ?>&to=<?php echo rawurlencode($today); ?>">Today</a>
    <a href="attendance_list.php?from=<?php echo rawurlencode(date('Y-m-01')); ?>&to=<?php echo rawurlencode($today); ?>">This Month</a>
    <a href="attendance_list.php">All</a>
</form>

<table>
<tr>
    <th>Date</th>
    <th>Staff</th>
    <th>Office IN</th>
    <th>Office OUT</th>
    <th>Home IN</th>
    <th>Home OUT</th>
    <th>Home Status</th>
    <th>Total</th>
    <th>OT</th>
    <th>Action</th>
</tr>

<?php
$rows = [];

$sql = "
    SELECT a.*, s.name
    FROM attendance a
    JOIN staff_account s ON s.id=a.staff_id
";
$params = [];
$types = '';
$w = [];

if ($from !== '' && $to !== '') {
    $w[] = "a.attendance_date BETWEEN ? AND ?";
    $types .= 'ss';
    $params[] = $from;
    $params[] = $to;
} elseif ($from !== '') {
    $w[] = "a.attendance_date >= ?";
    $types .= 's';
    $params[] = $from;
} elseif ($to !== '') {
    $w[] = "a.attendance_date <= ?";
    $types .= 's';
    $params[] = $to;
}

if ($w) {
    $sql .= " WHERE " . implode(' AND ', $w);
}

$sql .= " ORDER BY a.attendance_date DESC, a.in_time DESC, a.id DESC";

$st = mysqli_prepare($conn, $sql);
if ($st) {
    if ($types !== '') {
        mysqli_stmt_bind_param($st, $types, ...$params);
    }
    mysqli_stmt_execute($st);
    $rows = sb_stmt_fetch_all_assoc($st);
    mysqli_stmt_close($st);
}

$counts = [];
foreach ($rows as $r) {
  $d = (string)$r['attendance_date'];
  if (!isset($counts[$d])) $counts[$d] = 0;
  $counts[$d]++;
}

$cur = '';
foreach ($rows as $r) {
  $d = (string)$r['attendance_date'];
  if ($d !== $cur) {
    $cur = $d;
    $c = (int)($counts[$d] ?? 0);
    echo '<tr class="date-row"><td colspan="10" style="text-align:left;font-weight:bold;background:#eef5ff;">📅 '.htmlspecialchars($d).' — Total: '.$c.' logs</td></tr>';
  }

  $qs = '';
  if ($from !== '') $qs .= '&from=' . rawurlencode($from);
  if ($to !== '')   $qs .= '&to=' . rawurlencode($to);
?>
<tr>
<td><?php echo htmlspecialchars((string)$r['attendance_date']); ?></td>
<td><?php echo htmlspecialchars((string)$r['name']); ?></td>
<td><?php echo !empty($r['in_time'])  ? htmlspecialchars(sb_fmt_ampm((string)$r['in_time']))  : '-'; ?></td>
<td><?php echo !empty($r['out_time']) ? htmlspecialchars(sb_fmt_ampm((string)$r['out_time'])) : '-'; ?></td>
<td><?php echo !empty($r['home_in_time'])  ? htmlspecialchars(sb_fmt_ampm((string)$r['home_in_time']))  : '-'; ?></td>
<td><?php echo !empty($r['home_out_time']) ? htmlspecialchars(sb_fmt_ampm((string)$r['home_out_time'])) : '-'; ?></td>
<td><span class="pill2"><?php echo htmlspecialchars(sb_home_status($r)); ?></span></td>
<td><?php echo htmlspecialchars((string)$r['total_hours']); ?></td>
<td><?php echo htmlspecialchars((string)$r['overtime_hours']); ?></td>
<td class="action">
    <a class="edit" href="attendance_edit.php?id=<?php echo (int)$r['id']; ?><?php echo $qs; ?>">Edit</a>
    <?php if ($is_admin) { ?>
    <a class="delete" href="?delete=<?php echo (int)$r['id']; ?><?php echo $qs; ?>"
       onclick="return confirm('Delete record?')">Delete</a>
    <?php } ?>
</td>
</tr>
<?php } ?>
</table>

<div class="back">
    <a href="dashboard.php">⬅ Back to Dashboard</a>
</div>

</div>
</body>
</html>
