<?php
// attendance_out.php (Upgraded)
// - mysqlnd-free (no mysqli_stmt_get_result)
// - Custom OUT time input
// - Bigger UI window + stylish buttons
// - Work/Comment required and stored

require_once 'db.php';
require_once 'schema_guard.php';
require_once 'auth.php';
require_once 'ui.php';

require_login();
require_permission($conn, 'attendance_out');

sb_ensure_attendance_schema($conn);

@date_default_timezone_set('Asia/Karachi');

$uid      = current_user_id();
$today    = date('Y-m-d');
$now_time = date('H:i:s');
$now_time_ampm = date('h:i A');

$msg  = '';
$type = 'warn';

// Linked staff
$staff_id = 0;
$staff_name = '';
$st = mysqli_prepare($conn, "SELECT id, name FROM staff_account WHERE user_id=? LIMIT 1");
if ($st) {
  mysqli_stmt_bind_param($st, "i", $uid);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st, $sid, $sname);
  if (mysqli_stmt_fetch($st)) {
    $staff_id = (int)$sid;
    $staff_name = (string)$sname;
  }
  mysqli_stmt_close($st);
}


// Admin can choose any staff if not linked
$admin_pick_staff = false;
if ($staff_id <= 0 && is_admin()) {
  $admin_pick_staff = true;
  $picked = (int)($_POST['staff_id'] ?? ($_SESSION['pick_staff_id'] ?? 0));
  if ($picked > 0) {
    $ps = mysqli_prepare($conn, "SELECT id, name FROM staff_account WHERE id=? LIMIT 1");
    if ($ps) {
      mysqli_stmt_bind_param($ps, "i", $picked);
      mysqli_stmt_execute($ps);
      mysqli_stmt_bind_result($ps, $pid, $pname);
      if (mysqli_stmt_fetch($ps)) {
        $staff_id = (int)$pid;
        $staff_name = (string)$pname;
        $_SESSION['pick_staff_id'] = $staff_id;
      }
      mysqli_stmt_close($ps);
    }
  }
}

if ($staff_id <= 0 && !is_admin()) {
  sb_page_header('Attendance OUT');
  echo '<div class="msg err">Your user is not linked with staff account. Admin please set <b>staff_account.user_id</b> = <b>users.id</b>.</div>';
  echo '<div class="actions" style="margin-top:10px"></div>';
  sb_page_footer();
  exit;
}

$csrf = csrf_token();

/** Convert HH:MM input to HH:MM:SS, return '' on invalid */
function sb_hhmm_to_hhmmss(string $hhmm): string {
  $hhmm = trim($hhmm);
  if (!preg_match('/^([01]\d|2[0-3]):([0-5]\d)$/', $hhmm, $m)) return '';
  return $m[0] . ':00';
}

if (isset($_POST['checkout'])) {
  if (!csrf_validate($_POST['csrf'] ?? '')) {
    $msg = 'Security error (CSRF). Please refresh.';
    $type = 'err';
  } else {
    $work = trim((string)($_POST['out_work'] ?? ''));
    $out_hhmm = (string)($_POST['out_hhmm'] ?? '');
    $custom_out = sb_hhmm_to_hhmmss($out_hhmm);

    if ($custom_out === '') {
      $msg = 'Please select a valid OUT time (HH:MM).';
      $type = 'warn';
    } elseif ($work === '') {
      $msg = 'Please write work/comment before OUT.';
      $type = 'warn';
    } else {

      // Find today's record (mysqlnd-free)
      $att_id = 0;
      $in_time = '';
      $out_time_db = '';

      $q = mysqli_prepare($conn, "SELECT id, in_time, out_time FROM attendance WHERE staff_id=? AND attendance_date=? LIMIT 1");
      if ($q) {
        mysqli_stmt_bind_param($q, "is", $staff_id, $today);
        mysqli_stmt_execute($q);
        mysqli_stmt_bind_result($q, $rid, $rin, $rout);
        if (mysqli_stmt_fetch($q)) {
          $att_id = (int)$rid;
          $in_time = (string)$rin;
          $out_time_db = (string)$rout;
        }
        mysqli_stmt_close($q);
      }

      if ($att_id <= 0) {
        $msg = 'You are not checked IN today. Please IN first.';
        $type = 'warn';
      } elseif (!empty($out_time_db)) {
        $msg = 'Already checked OUT today.';
        $type = 'warn';
      } elseif (empty($in_time)) {
        $msg = 'IN time missing. Contact admin.';
        $type = 'err';
      } else {

        // Calculate hours using custom OUT time
        $in_ts  = strtotime($today . ' ' . $in_time);
        $out_ts = strtotime($today . ' ' . $custom_out);

        if ($in_ts === false || $out_ts === false || $out_ts <= $in_ts) {
          $msg = 'Invalid time. OUT time must be greater than IN time.';
          $type = 'err';
        } else {
          $hours = round(($out_ts - $in_ts) / 3600, 2);
          $overtime = ($hours > 8) ? round(($hours - 8), 2) : 0;

          // Store in both out_work and out_note to keep compatibility
          $u = mysqli_prepare($conn, "UPDATE attendance
              SET out_time=?, total_hours=?, overtime_hours=?,
                  out_work=?, out_note=?,
                  updated_by=?, updated_at=NOW()
              WHERE id=? LIMIT 1");

          if (!$u) {
            $msg = 'Server error (prepare failed).';
            $type = 'err';
          } else {
            mysqli_stmt_bind_param($u, "sddssii", $custom_out, $hours, $overtime, $work, $work, $uid, $att_id);
            if (mysqli_stmt_execute($u)) {
              $msg = '✅ Attendance OUT marked. Total Hours: ' . (string)$hours;
              $type = 'ok';
            } else {
              $msg = '❌ Update failed: ' . htmlspecialchars(mysqli_stmt_error($u));
              $type = 'err';
            }
            mysqli_stmt_close($u);
          }
        }
      }
    }
  }
}

sb_page_header('Attendance OUT');

// Make the card bigger + light theme (local, safe)
?>
<style>
:root{--ink:#0f172a;--muted:#475569;--stroke:rgba(15,23,42,.10);--shadow:0 18px 44px rgba(2,6,23,.16);--r:22px;}
body{
  margin:0;
  font-family:system-ui,Arial;
  color:var(--ink);
  background-color:#e9eef6;
  background-image:
    radial-gradient(1200px 520px at 15% 10%, rgba(37,99,235,.22), transparent 60%),
    radial-gradient(900px 420px at 85% 15%, rgba(71,85,105,.30), transparent 60%),
    radial-gradient(1100px 520px at 55% 95%, rgba(59,130,246,.16), transparent 65%),
    radial-gradient(760px 420px at 10% 90%, rgba(148,163,184,.40), transparent 60%);
}
/* hide reset links (header) */
a[href*="forgot_password.php"], a[href*="reset"]{display:none!important}

.sb-card{
  max-width: 980px;
  margin: 18px auto 22px;
  padding: 18px;
  border-radius: var(--r);
  background: rgba(255,255,255,.78);
  backdrop-filter: blur(10px);
  border:1px solid rgba(255,255,255,.45);
  box-shadow: var(--shadow);
}
.sb-title{font-size:24px;font-weight:1100;letter-spacing:.2px;margin:0 0 10px 0}
.sb-meta{
  display:flex;gap:10px;flex-wrap:wrap;align-items:center;
  margin:8px 0 14px 0;
}
.pill{
  padding:10px 14px;border-radius:999px;
  background:rgba(255,255,255,.72);
  border:1px solid var(--stroke);
  box-shadow: 0 10px 20px rgba(2,6,23,.08);
  font-weight:1000;
}
.pill small{color:var(--muted);font-weight:900}

.sb-panel{
  position:relative;
  border-radius: 20px;
  padding: 18px;
  background: rgba(255,255,255,.70);
  border:1px solid rgba(15,23,42,.10);
  overflow:hidden;
}
.sb-panel:before{
  content:"";
  position:absolute;inset:-2px;
  background-image:
    radial-gradient(circle at 1px 1px, rgba(15,23,42,.08) 1px, transparent 1.9px),
    radial-gradient(720px 260px at 12% 0%, rgba(37,99,235,.22), transparent 60%),
    radial-gradient(680px 260px at 90% 10%, rgba(71,85,105,.20), transparent 60%),
    radial-gradient(650px 280px at 40% 100%, rgba(59,130,246,.14), transparent 60%);
  background-size: 18px 18px, auto, auto, auto;
  opacity:.90;
  pointer-events:none;
}
.sb-panel > *{position:relative;z-index:1}

.sb-row{display:flex;gap:14px;flex-wrap:wrap;align-items:end;}
.sb-field{flex:1;min-width:260px;}
.sb-field label{display:block;font-size:13px;color:var(--muted);margin:0 0 6px 2px;font-weight:1000;}
.sb-field input[type=time], .sb-field input[type=text], .sb-field select, textarea{
  width:100%;
  padding:14px 16px;
  border:1px solid rgba(120,140,160,.30);
  border-radius:16px;
  font-size:16px;
  outline:none;
  background:rgba(255,255,255,.88);
  box-shadow: 0 8px 16px rgba(2,6,23,.06) inset;
}
textarea{min-height:120px;resize:vertical}
.sb-actions,.sb-cta{display:flex;gap:12px;flex-wrap:wrap;margin-top:16px}
.btn.big{
  padding:14px 18px;border-radius:18px;font-size:16px;font-weight:1100;letter-spacing:.2px;border:0;cursor:pointer;
  box-shadow:0 14px 28px rgba(2,6,23,.12);
}
.btn.primary.big{background:linear-gradient(135deg, rgba(37,99,235,.95), rgba(71,85,105,.70));color:#fff}
.btn.danger.big{background:linear-gradient(135deg, rgba(239,68,68,.92), rgba(71,85,105,.70));color:#fff}
.btn.out.big{background:linear-gradient(135deg, rgba(71,85,105,.92), rgba(37,99,235,.55));color:#fff;text-decoration:none;display:inline-flex;align-items:center;justify-content:center}
.sb-note{font-size:12px;color:var(--muted);margin-top:10px;font-weight:800}

.msg{max-width:980px;margin:10px auto 0;padding:0 12px}
.msg .box{padding:12px 14px;border-radius:18px;background:rgba(255,255,255,.86);border:1px solid var(--stroke)}
.msg .ok{border-color:rgba(34,197,94,.45)}
.msg .warn{border-color:rgba(250,204,21,.55)}
.msg .err{border-color:rgba(239,68,68,.45);color:#8a1f1f}
</style>

<div class="sb-card">
  </div>

  <div class="sb-meta">
    Date: <b><?php echo htmlspecialchars($today); ?></b>
    &nbsp; | &nbsp; Current Time: <b><?php echo htmlspecialchars($now_time_ampm); ?></b>
    <?php if ($staff_name !== '') { ?>
      &nbsp; | &nbsp; Staff: <b><?php echo htmlspecialchars($staff_name); ?></b>
    <?php } ?>
  </div>

  <?php if ($msg !== '') { ?>
    <div class="msg <?php echo htmlspecialchars($type); ?>"><?php echo htmlspecialchars($msg); ?></div>
  <?php } ?>

  <div class="sb-panel">
<form method="post" autocomplete="off">
    <input type="hidden" name="csrf" value="<?php echo htmlspecialchars($csrf); ?>">

<?php if (!empty($admin_pick_staff)) { ?>
  <label>Select Staff (Admin)</label>
  <select name="staff_id" required>
    <option value="">-- Select Staff --</option>
    <?php
      $rs = mysqli_query($conn, "SELECT id, name FROM staff_account ORDER BY name");
      $cur = (int)$staff_id;
      if ($rs) {
        while ($r = mysqli_fetch_assoc($rs)) {
          $id = (int)$r['id'];
          $nm = (string)$r['name'];
          $sel = ($id === $cur) ? 'selected' : '';
          echo '<option value="'.htmlspecialchars((string)$id).'" '.$sel.'>'.htmlspecialchars($nm).' (ID '.$id.')</option>';
        }
        mysqli_free_result($rs);
      }
    ?>
  </select>
  <div class="sb-meta" style="margin-top:6px">Admin: select staff, then mark OUT.</div>
<?php } ?>


    <div class="sb-row">
      <div>
        <label>OUT Time (Editable)</label>
        <input type="time" name="out_hhmm" required
               value="<?php
                 $val = (string)($_POST['out_hhmm'] ?? substr($now_time,0,5));
                 echo htmlspecialchars($val);
               ?>">
      </div>
      <div>
        <label>Today</label>
        <input type="text" value="<?php echo htmlspecialchars($today); ?>" readonly>
      </div>
    </div>

    <label style="margin-top:10px">Work / Comment (Report)</label>
    <textarea name="out_work" rows="4" placeholder="Aaj maine kya kaam kiya (2-3 lines)" required><?php
      echo htmlspecialchars((string)($_POST['out_work'] ?? ''));
    ?></textarea>

    <div class="sb-cta">
      <button class="btn big danger" type="submit" name="checkout">🔴 Check OUT</button>
      <a class="btn big" href="dashboard.php">✅ Done / Dashboard</a>
    </div>
  </form>
</div>

<?php sb_page_footer();
