<?php
require_once 'db.php';
require_once 'auth.php';
require_once 'schema_guard.php';
require_once 'ui.php';

require_login();

// Admin can view all staff reports. Staff can view only their own report.
sb_ensure_attendance_schema($conn);

@date_default_timezone_set('Asia/Karachi');


$uid   = current_user_id();
$admin = is_admin();

// If staff, force staff_id to own and hide any attempt to view others
if (!$admin) {
  $my_staff_id = 0;
  $q = mysqli_prepare($conn, "SELECT id FROM staff_account WHERE user_id=? LIMIT 1");
  if ($q) {
    mysqli_stmt_bind_param($q, "i", $uid);
    mysqli_stmt_execute($q);
    mysqli_stmt_bind_result($q, $my_staff_id);
    mysqli_stmt_fetch($q);
    mysqli_stmt_close($q);
  }
  $staff_id = (int)$my_staff_id;
}


$records = [];
$totals = [
  'office' => 0.0,
  'overtime' => 0.0,
  'home' => 0.0,
  'sat' => 0.0,
  'grand' => 0.0,
];

$staff_id = $admin ? (int)($_POST['staff_id'] ?? 0) : (int)($staff_id ?? 0);
$month_val = (string)($_POST['month'] ?? date('Y-m'));

if (isset($_POST['search']) && $staff_id > 0) {
  $month_start = $month_val . '-01';
  $month_end = date('Y-m-d', strtotime($month_start . ' +1 month'));

  $sql = "SELECT attendance_date, in_time, out_time, total_hours, overtime_hours, home_hours, saturday_hours, out_work
          FROM attendance
          WHERE staff_id=? AND attendance_date >= ? AND attendance_date < ?
          ORDER BY attendance_date ASC";
  $st = mysqli_prepare($conn, $sql);
  if ($st) {
    mysqli_stmt_bind_param($st, 'iss', $staff_id, $month_start, $month_end);
    mysqli_stmt_execute($st);
    // mysqlnd-free fetch (NO mysqli_stmt_get_result)
    mysqli_stmt_bind_result($st, $att_date, $in_time, $out_time, $total_hours, $overtime_hours, $home_hours, $saturday_hours, $out_work);
    while (mysqli_stmt_fetch($st)) {
      $r = [
        'attendance_date' => (string)$att_date,
        'in_time' => (string)$in_time,
        'out_time' => (string)$out_time,
        'total_hours' => $total_hours,
        'overtime_hours' => $overtime_hours,
        'home_hours' => $home_hours,
        'saturday_hours' => $saturday_hours,
        'out_work' => (string)$out_work,
      ];

      $office = (float)($r['total_hours'] ?? 0);
      $ot     = (float)($r['overtime_hours'] ?? 0);
      $home   = (float)($r['home_hours'] ?? 0);
      $sat    = (float)($r['saturday_hours'] ?? 0);
      $final  = $office + $ot + $home + $sat;

      $r['final_total'] = $final;
      $records[] = $r;

      $totals['office'] += $office;
      $totals['overtime'] += $ot;
      $totals['home'] += $home;
      $totals['sat'] += $sat;
      $totals['grand'] += $final;
    }
    mysqli_stmt_close($st);
  }
}

// Staff dropdown
$staff_rows = [];
if (is_admin()) {
  $q = mysqli_query($conn, 'SELECT id, name FROM staff_account ORDER BY name ASC');
  while ($q && ($r = mysqli_fetch_assoc($q))) $staff_rows[] = $r;
} else {
  $uid = current_user_id();
  $st = mysqli_prepare($conn, 'SELECT id, name FROM staff_account WHERE user_id=? LIMIT 1');
  if ($st) {
    mysqli_stmt_bind_param($st, 'i', $uid);
    mysqli_stmt_execute($st);
    mysqli_stmt_bind_result($st, $sid, $sn);
    if (mysqli_stmt_fetch($st)) $staff_rows[] = ['id'=>$sid,'name'=>$sn];
    mysqli_stmt_close($st);
  }
}

sb_page_header('Monthly Attendance Report');
?>

<style>
  .readonly{padding:10px 12px;border:1px solid rgba(15,23,42,.15);border-radius:12px;background:#f8fafc;}
.filters{display:grid;grid-template-columns:1fr 1fr auto;gap:12px;align-items:end;}
  @media(max-width:720px){.readonly{padding:10px 12px;border:1px solid rgba(15,23,42,.15);border-radius:12px;background:#f8fafc;}
.filters{grid-template-columns:1fr;}}
  table{width:100%;border-collapse:separate;border-spacing:0;margin-top:12px;}
  th,td{padding:10px;border-bottom:1px solid var(--stroke);text-align:left;font-size:13.5px;vertical-align:top;}
  th{font-weight:1000;color:var(--muted);background:rgba(37,99,235,.06);} 
  tr:hover td{background:rgba(2,6,23,.02);} 
  .sum{display:flex;gap:10px;flex-wrap:wrap;margin-top:12px;}
  .pill{padding:8px 10px;border-radius:999px;border:1px solid var(--stroke);background:#fff;font-weight:1000;font-size:12.5px;}
</style>

<form method="post" autocomplete="off" class="filters">
  <?php if ($admin): ?>
    <div>
      <label>Staff</label>
      <select name="staff_id" required>
        <option value="">Select Staff</option>
        <?php foreach ($staff_rows as $s): ?>
          <option value="<?php echo (int)$s['id']; ?>" <?php echo ((int)$staff_id===(int)$s['id']?'selected':''); ?>>
            <?php echo htmlspecialchars($s['name']); ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>
  <?php else: ?>
    <?php
      $my_name = $staff_rows[0]['name'] ?? 'My';
      $my_id = (int)($staff_rows[0]['id'] ?? 0);
    ?>
    <input type="hidden" name="staff_id" value="<?php echo (int)$my_id; ?>">
    <div>
      <label>Staff</label>
      <div class="readonly"><?php echo htmlspecialchars($my_name); ?></div>
    </div>
  <?php endif; ?>

  <div>
    <label>Month</label>
    <input type="month" name="month" value="<?php echo htmlspecialchars($month_val); ?>" required>
  </div>
  <div>
    <button class="btn primary" type="submit" name="search">Search</button>
  </div>
</form>

<?php if (isset($_POST['search']) && $staff_id > 0) { ?>
  <div class="sum">
    <div class="pill">Office: <?php echo number_format($totals['office'],2); ?></div>
    <div class="pill">Overtime: <?php echo number_format($totals['overtime'],2); ?></div>
    <div class="pill">Home: <?php echo number_format($totals['home'],2); ?></div>
    <div class="pill">Saturday: <?php echo number_format($totals['sat'],2); ?></div>
    <div class="pill">Grand: <?php echo number_format($totals['grand'],2); ?></div>
  </div>

  <table>
    <thead>
      <tr>
        <th>Date</th>
        <th>IN</th>
        <th>OUT</th>
        <th>Office</th>
        <th>OT</th>
        <th>Home</th>
        <th>Sat</th>
        <th>Total</th>
        <th>OUT Comment</th>
      </tr>
    </thead>
    <tbody>
      <?php foreach ($records as $r): ?>
        <tr>
          <td><?php echo htmlspecialchars($r['attendance_date']); ?></td>
          <td><?php echo htmlspecialchars((string)($r['in_time'] ?? '')); ?></td>
          <td><?php echo htmlspecialchars((string)($r['out_time'] ?? '')); ?></td>
          <td><?php echo number_format((float)($r['total_hours'] ?? 0),2); ?></td>
          <td><?php echo number_format((float)($r['overtime_hours'] ?? 0),2); ?></td>
          <td><?php echo number_format((float)($r['home_hours'] ?? 0),2); ?></td>
          <td><?php echo number_format((float)($r['saturday_hours'] ?? 0),2); ?></td>
          <td><b><?php echo number_format((float)($r['final_total'] ?? 0),2); ?></b></td>
          <td><?php echo nl2br(htmlspecialchars((string)($r['out_work'] ?? ''))); ?></td>
        </tr>
      <?php endforeach; ?>
    </tbody>
  </table>
<?php } ?>

<?php sb_page_footer();
