<?php
require_once 'sb_session.php';

/* auth.php — Study Bridge (Unified Auth Guard)
   - Dedicated session cookie (SBSESSID)
   - Main keys: user_id, username, role
   - Backward compatible with admin/admin_id and sb_* keys
*/

// Backward compatibility: if admin keys exist, mirror into user keys
if (empty($_SESSION['user_id']) && !empty($_SESSION['admin_id'])) {
  $_SESSION['user_id']  = $_SESSION['admin_id'];
  $_SESSION['username'] = $_SESSION['admin'] ?? ($_SESSION['username'] ?? '');
  $_SESSION['role']     = $_SESSION['role'] ?? 'admin';
}

// Backward compatibility: if sb_* keys exist, mirror into user keys
if (empty($_SESSION['user_id']) && !empty($_SESSION['sb_user_id'])) {
  $_SESSION['user_id']  = $_SESSION['sb_user_id'];
  $_SESSION['username'] = $_SESSION['sb_username'] ?? ($_SESSION['username'] ?? '');
  $_SESSION['role']     = $_SESSION['sb_role'] ?? ($_SESSION['role'] ?? 'staff');
}

function require_login(){
  if (empty($_SESSION['user_id']) || empty($_SESSION['username'])) {
    header("Location: login.php");
    exit();
  }
}

function current_user_id(){ return (int)($_SESSION['user_id'] ?? 0); }
function current_username(){ return (string)($_SESSION['username'] ?? ''); }
function current_role(){ return (string)($_SESSION['role'] ?? ''); }

function is_admin(){
  return (current_role() === 'admin');
}

/**
 * Permissions helper (user_permissions table).
 * Admin bypass.
 */
function sb_user_permissions(mysqli $conn, int $uid): array {
  $perms = [
    'staff_manage'   => 0,
    // Attendance actions should work for staff by default.
    // If you want to restrict later, insert a user_permissions row and set these to 0.
    'attendance_in'  => 1,
    'attendance_out' => 1,
    'home_hours'     => 1,
    'reports'        => 0,
  ];
  if ($uid <= 0) return $perms;

  $st = mysqli_prepare($conn, "SELECT staff_manage, attendance_in, attendance_out, home_hours, reports FROM user_permissions WHERE user_id=? LIMIT 1");
if (!$st) return $perms;
mysqli_stmt_bind_param($st, "i", $uid);
mysqli_stmt_execute($st);

// mysqlnd-free fetch
mysqli_stmt_bind_result($st, $p_staff_manage, $p_att_in, $p_att_out, $p_home, $p_reports);
$row = null;
if (mysqli_stmt_fetch($st)) {
  $row = [
    'staff_manage'   => (int)$p_staff_manage,
    'attendance_in'  => (int)$p_att_in,
    'attendance_out' => (int)$p_att_out,
    'home_hours'     => (int)$p_home,
    'reports'        => (int)$p_reports
  ];
}
mysqli_stmt_close($st);


  if ($row) {
    foreach ($perms as $k => $v) {
      if (isset($row[$k])) $perms[$k] = (int)$row[$k] ? 1 : 0;
    }
  }
  return $perms;
}

function require_permission(mysqli $conn, string $perm_key): void {
  require_login();
  if (is_admin()) return;
  $uid = current_user_id();
  $perms = sb_user_permissions($conn, $uid);
  if (empty($perms[$perm_key])) {
    header("Location: dashboard.php?e=permission_denied");
    exit();
  }
}

/* simple csrf */
function csrf_token(){
  if (empty($_SESSION['csrf'])) {
    $_SESSION['csrf'] = bin2hex(random_bytes(16));
  }
  return $_SESSION['csrf'];
}

function csrf_validate($token){
  return isset($_SESSION['csrf']) && is_string($token) && hash_equals($_SESSION['csrf'], $token);
}
