<?php
require_once 'sb_session.php';
require_once 'db.php';
require_once 'schema_guard.php';

// Auth (admin or user)
// NOTE: Different installs store different session keys. Keep this permissive so staff users are not blocked.
$has_login = !empty($_SESSION['user_id'])
    || !empty($_SESSION['admin_id'])
    || !empty($_SESSION['sb_user_id'])
    || !empty($_SESSION['username'])
    || !empty($_SESSION['sb_username'])
    || !empty($_SESSION['name'])
    || !empty($_SESSION['full_name'])
    || !empty($_SESSION['admin']);

if (!$has_login) {
    header("Location: login.php");
    exit();
}

date_default_timezone_set("Asia/Karachi");

// Ensure required tables/columns exist (safe for existing installs)
sb_ensure_permissions_schema($conn);
sb_ensure_attendance_schema($conn);
$today = date("Y-m-d");
$today_display = date("d M Y");
$current_time = date("h:i A");

// Time formatter: 12-hour AM/PM
function sb_fmt_ampm($timeStr): string {
    $timeStr = trim((string)$timeStr);
    if ($timeStr === '' || strtoupper($timeStr) === 'NULL') return '';
    $ts = strtotime($timeStr);
    if ($ts === false) return $timeStr;
    return date('h:i A', $ts);
}
// Resolve display name (avoid showing literal 'admin' when possible)
function sb_table_has_column($conn, $table, $column){
    $table = mysqli_real_escape_string($conn, $table);
    $column = mysqli_real_escape_string($conn, $column);
    $sql = "SELECT 1 FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = '$table' AND COLUMN_NAME = '$column' LIMIT 1";
    $q = mysqli_query($conn, $sql);
    return ($q && mysqli_num_rows($q) > 0);
}
function sb_safe_fetch_name($conn, $table, $idCol, $nameCol, $id){
    $id = (int)$id;
    if ($id <= 0) return '';
    $tableEsc = preg_replace('/[^a-zA-Z0-9_]/','', $table);
    $idColEsc = preg_replace('/[^a-zA-Z0-9_]/','', $idCol);
    $nameColEsc = preg_replace('/[^a-zA-Z0-9_]/','', $nameCol);
    $sql = "SELECT `$nameColEsc` AS n FROM `$tableEsc` WHERE `$idColEsc` = $id LIMIT 1";
    $q = mysqli_query($conn, $sql);
    if ($q && ($r = mysqli_fetch_assoc($q))) return (string)($r['n'] ?? '');
    return '';
}

// Format time to 12-hour AM/PM for dashboard display
function sb_fmt_time_12hr($t): string {
    $t = trim((string)$t);
    if ($t === '' || $t === '00:00:00') return '';
    $ts = strtotime($t);
    if ($ts === false) return $t;
    return date('h:i A', $ts);
}

$displayUser = (string)($_SESSION['name'] ?? $_SESSION['full_name'] ?? $_SESSION['username'] ?? $_SESSION['sb_username'] ?? $_SESSION['admin'] ?? '');
$role = (string)($_SESSION['role'] ?? $_SESSION['sb_role'] ?? '');

// Try to resolve admin/user name from DB if we only have 'admin' or empty
if ($displayUser === '' || strtolower($displayUser) === 'admin') {
    $resolved = '';
    if (!empty($_SESSION['user_id'])) {
        if (sb_table_has_column($conn, 'users', 'name')) $resolved = sb_safe_fetch_name($conn, 'users', 'id', 'name', (int)$_SESSION['user_id']);
        if ($resolved==='' && sb_table_has_column($conn, 'users', 'username')) $resolved = sb_safe_fetch_name($conn, 'users', 'id', 'username', (int)$_SESSION['user_id']);
    }
    if ($resolved==='' && !empty($_SESSION['admin_id'])) {
        if (sb_table_has_column($conn, 'admins', 'name')) $resolved = sb_safe_fetch_name($conn, 'admins', 'id', 'name', (int)$_SESSION['admin_id']);
        if ($resolved==='' && sb_table_has_column($conn, 'admin', 'name')) $resolved = sb_safe_fetch_name($conn, 'admin', 'id', 'name', (int)$_SESSION['admin_id']);
        if ($resolved==='' && sb_table_has_column($conn, 'admin_users', 'name')) $resolved = sb_safe_fetch_name($conn, 'admin_users', 'id', 'name', (int)$_SESSION['admin_id']);
        if ($resolved==='' && sb_table_has_column($conn, 'admin_users', 'username')) $resolved = sb_safe_fetch_name($conn, 'admin_users', 'id', 'username', (int)$_SESSION['admin_id']);
    }
    if ($resolved !== '') $displayUser = $resolved;
    if ($displayUser === '' || strtolower($displayUser) === 'admin') $displayUser = 'Administrator';
}

// Staff list
// Hide ghost "ADMIN" row if admin user was deleted (common in some installs).
// Also prefer active staff if a status column exists.
$staff_sql = "SELECT id, name, designation FROM staff_account";
if (sb_table_has_column($conn, 'staff_account', 'status')) {
    $staff_sql .= " WHERE status='Active'";
    $staff_sql .= " AND UPPER(TRIM(name)) <> 'ADMIN'";
} else {
    $staff_sql .= " WHERE UPPER(TRIM(name)) <> 'ADMIN'";
}
$staff_sql .= " ORDER BY name ASC";
$staff_q = mysqli_query($conn, $staff_sql);

// Today's attendance
$att_map = [];
$hasHomeHours = sb_table_has_column($conn, 'attendance', 'home_hours');
$att_sql = $hasHomeHours
    ? "SELECT staff_id, user_id, in_time, out_time, total_hours, home_hours FROM attendance WHERE attendance_date = '$today'"
    : "SELECT staff_id, in_time, out_time, total_hours FROM attendance WHERE attendance_date = '$today'";
$att_q = mysqli_query($conn, $att_sql);
if ($att_q) {
    while ($r = mysqli_fetch_assoc($att_q)) {
        $att_map[(int)$r['staff_id']] = $r;
    }
}
?>
<!DOCTYPE html>
<html>
<head>
<title>Study Bridge | Dashboard</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<style>
  :root{
    /* LIGHT professional palette */
    --bg:#f5f7fb;
    --card:#ffffff;
    --card2:#fbfcff;

    --text:#0f172a;
    --muted:#475569;

    --pri:#2563eb;      /* blue */
    --pri2:#1d4ed8;     /* deeper blue */
    --acc:#06b6d4;      /* cyan */
    --stroke: rgba(15,23,42,.10);
    --stroke2: rgba(15,23,42,.16);

    --shadow: 0 14px 38px rgba(15,23,42,.10);
    --shadow2: 0 10px 24px rgba(15,23,42,.08);

    --radius: 18px;
  }

  *{box-sizing:border-box;}
  body{
    margin:0;
    font-family: ui-sans-serif, system-ui, -apple-system, Segoe UI, Roboto, Arial, "Noto Sans", "Liberation Sans", sans-serif;
    background:
      radial-gradient(900px 500px at 12% 8%, rgba(37,99,235,.18), transparent 60%),
      radial-gradient(900px 600px at 88% 22%, rgba(6,182,212,.16), transparent 60%),
      linear-gradient(180deg, #ffffff, var(--bg));
    color:var(--text);
    min-height:100vh;
  }

  /* TOPBAR */
  .topbar{
    position:sticky; top:0; z-index:10;
    display:flex; align-items:center; justify-content:space-between;
    padding:14px 18px;
    background: rgba(255,255,255,.88);
    backdrop-filter: blur(10px);
    border-bottom:1px solid var(--stroke);
  }

  /* Moving 6mm watercolor rainbow line under top bar */
  .topbar::after{
    content:'';
    position:absolute;
    left:10px; right:10px;
    bottom:0px;
    height:14px; /* smaller */
    background:
      linear-gradient(90deg,
        rgba(239,68,68,.90),
        rgba(245,158,11,.90),
        rgba(234,179,8,.90),
        rgba(34,197,94,.90),
        rgba(59,130,246,.90),
        rgba(168,85,247,.90),
        rgba(236,72,153,.90),
        rgba(239,68,68,.90)
      );
    background-size: 220% 100%;
    opacity:.9;
    border-top-left-radius:999px;
    border-top-right-radius:999px;
    border-bottom-left-radius:18px;
    border-bottom-right-radius:18px;
    animation: rainbowSlide 6s ease-in-out infinite alternate;
  }
  @keyframes rainbowSlide{
    0%{ background-position: 0% 50%; }
    100%{ background-position: 100% 50%; }
  }

  .top-left{
    display:flex; align-items:center; gap:10px;
    min-width:220px;
  }
  .menu-btn{
    display:none;
    width:42px; height:42px;
    border-radius:12px;
    border:1px solid var(--stroke2);
    background: #fff;
    color:var(--text);
    cursor:pointer;
    box-shadow: var(--shadow2);
  }
  .pill{
    display:inline-flex; align-items:center; gap:8px;
    padding:9px 12px;
    border-radius:999px;
    border:1px solid var(--stroke);
    background: #fff;
    color: var(--muted);
    font-size:12.5px;
    white-space:nowrap;
    box-shadow: var(--shadow2);
  }
  .dot{
    width:8px; height:8px; border-radius:999px;
    background: var(--acc);
    box-shadow: 0 0 0 4px rgba(6,182,212,.14);
  }

  /* ✅ ONE-LINE BIG BRAND */
  .brand{font-weight:1000;font-size:22px;letter-spacing:.6px;color:var(--pri);transform:translateY(-6px);}
  .brand span{
    background: linear-gradient(90deg, var(--pri), var(--acc));
    -webkit-background-clip:text;
    background-clip:text;
    color:transparent;
  }

  .top-right{
    min-width:220px;
    display:flex; flex-direction:column; align-items:flex-end; gap:8px;
  }
  .userline{
    font-size:12.5px;
    color: var(--muted);
    line-height:1.2;
  }
  .userline b{ color:var(--text); }

  .logout-btn{
    display:inline-flex;
    align-items:center;
    justify-content:center;
    padding:9px 14px;
    border-radius:12px;
    text-decoration:none;
    font-weight:900;
    color:#fff;
    border:1px solid rgba(37,99,235,.25);
    background: linear-gradient(135deg, var(--pri), var(--pri2));
    box-shadow: 0 12px 22px rgba(37,99,235,.22);
  }
  .logout-btn:hover{ filter:brightness(1.05); }

  /* LAYOUT */
  .layout{
    display:grid;
    grid-template-columns: 1fr;
    gap:16px;
    max-width: 100%;
    margin: 18px auto;
    padding: 0 18px 22px;
  }

  /* SIDEBAR */
  .sidebar{
    position:sticky; top:78px;
    height: calc(100vh - 96px);
    border:1px solid var(--stroke);
    border-radius: var(--radius);
    background: rgba(255,255,255,.86);
    box-shadow: var(--shadow);
    overflow:hidden;
  }
  .side-head{
    padding:16px 16px 12px;
    border-bottom:1px solid var(--stroke);
    background: linear-gradient(180deg, rgba(37,99,235,.10), rgba(255,255,255,0));
  }
  .side-title{ font-weight:1000; letter-spacing:.4px; }
  .side-sub{ margin-top:6px; font-size:12.5px; color:var(--muted); display:flex; gap:10px; flex-wrap:wrap; }
  .kbd{
    padding:4px 8px;
    border-radius:999px;
    border:1px solid var(--stroke);
    background: #fff;
  }

  .nav{
    padding:12px;
    display:flex;
    flex-direction:column;
    gap:10px;
  }
  .nav a{
    display:flex;
    align-items:center;
    justify-content:space-between;
    gap:10px;
    padding:12px 12px;
    border-radius:14px;
    text-decoration:none;
    color:var(--text);
    border:1px solid var(--stroke);
    background: #fff;
    transition: .15s ease;
    font-weight:900;
    font-size:13.5px;
    box-shadow: var(--shadow2);
  }
  .nav a:hover{
    transform: translateY(-1px);
    border-color: rgba(37,99,235,.28);
    box-shadow: 0 16px 30px rgba(15,23,42,.12);
  }

  /* BIG right-side navigation buttons (~1 inch height) */
  .nav .nav-btn{
    min-height:96px;
    padding:18px 14px;
    font-size:16px;
  }
  .nav .nav-left{ display:flex; align-items:center; gap:12px; }
  .nav .ico{ font-size:28px; line-height:1; }
  .nav .lbl{ font-weight:1000; letter-spacing:.2px; }
  .nav .nav-logout{ border-color: rgba(239,68,68,.35); }
  .nav .nav-logout:hover{ border-color: rgba(239,68,68,.65); }
  .nav-spacer{ height:8px; }

  /* Tooltip (shows name on hover) */
  .nav .nav-btn{ position:relative; }
  .nav .nav-btn:hover::after{
    content: attr(data-tip);
    position:absolute;
    right: 14px;
    top: -10px;
    transform: translateY(-100%);
    background: rgba(15,23,42,.92);
    color:#fff;
    padding:6px 10px;
    border-radius:10px;
    font-size:12.5px;
    font-weight:900;
    white-space:nowrap;
    box-shadow: 0 18px 35px rgba(15,23,42,.25);
  }
  .nav .nav-btn:hover::before{
    content:'';
    position:absolute;
    right: 26px;
    top: -10px;
    width:10px;
    height:10px;
    background: rgba(15,23,42,.92);
    transform: translateY(-50%) rotate(45deg);
    border-radius:2px;
  }
  .tag{
    font-size:11.5px;
    color: var(--muted);
    border:1px solid var(--stroke);
    background: var(--card2);
    padding:3px 8px;
    border-radius:999px;
    white-space:nowrap;
    font-weight:900;
  }

  /* CONTENT */
  .content{
    display:flex;
    flex-direction:column;
    gap:16px;
  }

  .hero{
    border:1px solid var(--stroke);
    border-radius: var(--radius);
    background: rgba(255,255,255,.86);
    box-shadow: var(--shadow);
    padding:16px;
    display:flex;
    align-items:center;
    justify-content:space-between;
    gap:12px;
  }
  .hero-left{ display:flex; flex-direction:column; gap:6px; }
  .hero-left .h1{ font-size:18px; font-weight:1000; letter-spacing:.2px; }
  .hero-left .h2{ font-size:13px; color:var(--muted); line-height:1.4; }
  .hero-right{ display:flex; gap:10px; flex-wrap:wrap; justify-content:flex-end; }
  .mini{
    border:1px solid var(--stroke);
    background: #fff;
    border-radius:14px;
    padding:10px 12px;
    min-width:160px;
    text-align:left;
    box-shadow: var(--shadow2);
  }
  .mini .k{ font-size:12.5px; color:var(--muted); }
  .mini .v{ font-weight:1000; margin-top:2px; font-size:14px; }

  /* ✅ STANDARDIZED CARD + TABLE TYPOGRAPHY */
  .card{
    border:1px solid var(--stroke);
    border-radius: var(--radius);
    background: #fff;
    box-shadow: var(--shadow);
    overflow:hidden;
  }
  .card-head{
    padding:14px 16px;
    display:flex;
    align-items:center;
    justify-content:space-between;
    gap:12px;
    background: linear-gradient(135deg, rgba(37,99,235,.96), rgba(6,182,212,.86));
    color:#fff;
    font-weight:1000;
    font-size:15px;
    letter-spacing:.2px;
  }
  .card-head small{
    font-weight:900;
    opacity:.95;
    font-size:12.5px;
  }

  
  /* STATUS CARD (Today staff status) - refreshed colors */
  .status-card{
    border-color: rgba(99,102,241,.22);
    background: linear-gradient(180deg, rgba(99,102,241,.08), rgba(255,255,255,.86));
  }
  .status-card .card-head{
    background: linear-gradient(135deg, rgba(99,102,241,.96), rgba(236,72,153,.92));
  }
  .status-card .card-head small{ opacity:.9; }
  .status-card .table-container{ border-color: rgba(99,102,241,.20); }
  .status-card tr:hover td{ background: rgba(99,102,241,.06); }

.table-container{ width:100%; overflow:auto; max-height: calc(100vh - 270px); }
  table{
    width:100%;
    border-collapse:collapse;
    min-width:980px;
    font-size:13.5px;          /* ✅ same base size */
    line-height:1.35;          /* ✅ same line height */
  }
  th, td{
    padding:12px 10px;         /* ✅ same padding */
    border-bottom:1px solid rgba(15,23,42,.10);
    text-align:center;
    white-space:nowrap;
  }
  th{
    position:sticky; top:0;
    background: linear-gradient(90deg, rgba(220,38,38,.92), rgba(37,99,235,.92), rgba(22,163,74,.92));
    color:#fff;
    font-weight:1000;
    z-index:1;
    font-size:13.5px;          /* ✅ same size */
  }
  tr:hover td{ background: rgba(37,99,235,.06); }

  .badge{
    display:inline-flex;
    align-items:center;
    justify-content:center;
    padding:6px 10px;
    border-radius:999px;
    font-weight:1000;
    font-size:12.5px;
    border:1px solid transparent;
  }
  .b-present{ background: rgba(34,197,94,.14); color:#166534; border-color: rgba(34,197,94,.35); }
  .b-absent{  background: rgba(239,68,68,.14); color:#991b1b; border-color: rgba(239,68,68,.35); }
  .b-pending{ background: rgba(245,158,11,.18); color:#92400e; border-color: rgba(245,158,11,.35); }

  .footer{
    margin-top:6px;
    text-align:center;
    color: var(--muted);
    font-size:12.5px;
  }
  .footer b{ color: rgba(15,23,42,.88); }


  .devby{
    position:fixed;
    left:12px;
    bottom:10px;
    font-size:12px;
    font-weight:900;
    color: rgba(71,85,105,.92);
    background: rgba(255,255,255,.80);
    border:1px solid var(--stroke);
    padding:6px 10px;
    border-radius:999px;
    box-shadow: var(--shadow2);
    z-index: 20;
  }

  /* Mobile */
  @media (max-width: 980px){
    .layout{ grid-template-columns: 1fr; }
    .sidebar{ position:relative; top:auto; height:auto; }
    .top-left{ min-width:auto; }
    .top-right{ min-width:auto; }
    .menu-btn{ display:inline-flex; align-items:center; justify-content:center; }
    .sidebar{ display:none; }
    body.sidebar-open .sidebar{ display:block; }
    .brand{font-weight:1000;font-size:22px;letter-spacing:.6px;color:var(--pri);transform:translateY(-6px);}
  }

  .logout-fixed{
    position:fixed;
    right:14px;
    bottom:12px;
    z-index:9999;
    padding:11px 14px;
    border-radius:14px;
    background:
      radial-gradient(circle at 20% 30%, rgba(255,255,255,.28) 0 2px, transparent 3px),
      radial-gradient(circle at 70% 60%, rgba(255,255,255,.22) 0 2px, transparent 3px),
      repeating-linear-gradient(45deg, rgba(255,255,255,.14) 0 2px, transparent 2px 7px),
      linear-gradient(90deg, rgba(220,38,38,.92), rgba(37,99,235,.92), rgba(22,163,74,.92));
    color:#fff;
    text-decoration:none;
    font-weight:900;
    box-shadow: 0 12px 24px rgba(15,23,42,.18);
    border:1px solid rgba(255,255,255,.22);
  }
  .logout-fixed:hover{opacity:.92}
  @media (max-width: 900px){
    .logout-fixed{right:10px;bottom:10px}
  }

  .formsbar{
    margin: 12px 0 0 0;
    background: rgba(255,255,255,.85);
    border:1px solid var(--stroke);
    border-radius: var(--radius);
    box-shadow: var(--shadow2);
    padding: 12px 12px 10px 12px;
  }
  .forms-title{font-weight:1000;color:var(--text);margin:0 0 10px 4px}
  .forms-row{
    display:flex;
    flex-wrap:wrap;
    gap:12px;
    justify-content:center;
  }
  .formbtn{
    width: 136px;  /* ~1.4 inch */
    height: 136px; /* ~1.4 inch */
    border-radius: 22px;
    border:1px solid rgba(15,23,42,.10);
    background: linear-gradient(180deg,#ffffff,#f7fbff);
    box-shadow: 0 14px 26px rgba(15,23,42,.08);
    text-decoration:none;
    color: var(--text);
    display:flex;
    flex-direction:column;
    align-items:center;
    justify-content:center;
    gap:10px;
    position:relative;
    user-select:none;
    transition:.15s transform, .15s box-shadow;
  }

  /* Moving light watercolor outline for form link buttons (9 colors) */
  .formbtn{ overflow:hidden; }
  .formbtn::before{
    content:"";
    position:absolute;
    inset:0;
    padding:2px;                 /* border thickness */
    border-radius:inherit;
    pointer-events:none;
    z-index:0;
    opacity:.88;

    /* thin traveling watercolor stroke around the border */
    background: conic-gradient(
      from 0deg,
      transparent 0 88%,
      color-mix(in srgb, var(--ring-color, rgba(59,130,246,.70)) 78%, white) 88% 92%,
      transparent 92% 100%
    );

    /* show only the border area (works in modern Chromium) */
    -webkit-mask: linear-gradient(#000 0 0) content-box, linear-gradient(#000 0 0);
    -webkit-mask-composite: xor;
            mask-composite: exclude;

    animation: formRingSpin var(--ring-speed, 18s) linear infinite;
    animation-delay: var(--ring-delay, 0s);
}
.formbtn > *{ position:relative; z-index:1; }

  @keyframes formRingSpin{
    to{ transform: rotate(360deg); }
  }

  /* Assign 9 soft colors (watercolor/light) */
  .forms-row .formbtn:nth-child(1){--ring-color: rgba(59,130,246,.70); --ring-speed: 18.5s; --ring-delay: -0.8s;}   /* blue */
  .forms-row .formbtn:nth-child(2){--ring-color: rgba(34,197,94,.70); --ring-speed: 22.0s; --ring-delay: -3.2s;}    /* green */
  .forms-row .formbtn:nth-child(3){--ring-color: rgba(239,68,68,.70); --ring-speed: 17.2s; --ring-delay: -1.7s;}    /* red */
  .forms-row .formbtn:nth-child(4){--ring-color: rgba(249,115,22,.70); --ring-speed: 24.6s; --ring-delay: -6.1s;}   /* orange */
  .forms-row .formbtn:nth-child(5){--ring-color: rgba(234,179,8,.70); --ring-speed: 21.4s; --ring-delay: -4.4s;}    /* yellow */
  .forms-row .formbtn:nth-child(6){--ring-color: rgba(236,72,153,.70); --ring-speed: 26.8s; --ring-delay: -2.5s;}   /* pink */
  .forms-row .formbtn:nth-child(7){--ring-color: rgba(128,0,32,.60); --ring-speed: 19.6s; --ring-delay: -5.8s;}     /* mehroon/maroon */
  .forms-row .formbtn:nth-child(8){--ring-color: rgba(168,85,247,.70); --ring-speed: 23.2s; --ring-delay: -0.3s;}   /* purple */
  .forms-row .formbtn:nth-child(9){--ring-color: rgba(34,197,94,.55); --ring-speed: 28.0s; --ring-delay: -7.6s;}    /* parrot (lighter green) */
  /* Stagger + non-matching motion */
  .forms-row .formbtn:nth-child(even)::before{ animation-direction: reverse; }



  .formbtn:hover{transform:translateY(-2px);box-shadow: 0 18px 32px rgba(15,23,42,.10)}
  .fi{font-size:44px;line-height:1}
  .fi img{width:54px;height:54px;object-fit:contain;filter: drop-shadow(0 8px 14px rgba(15,23,42,.10));}
  .ft{font-weight:1000;font-size:14px;letter-spacing:.2px}
  @media (max-width: 650px){
    .formbtn{width: 118px;height:118px;border-radius:20px}
    .fi{font-size:38px}
  }

  /* CAPS for dashboard */
  .brand,.forms-title,.ft,.card-head,.side-title,.nav-btn,.table-container,table,th,td{ text-transform: uppercase; letter-spacing: .4px; }

  
  /* Watercolor header background (applied to column headers only) */
  th{
    background-image:
      linear-gradient(90deg,
        rgba(239,68,68,.28),
        rgba(59,130,246,.28),
        rgba(34,197,94,.28),
        rgba(250,204,21,.22),
        rgba(236,72,153,.22),
        rgba(45,212,191,.22),
        rgba(168,85,247,.22),
        rgba(239,68,68,.28)
      );
    background-size: 240% 100%;
    animation: thWave 7s ease-in-out infinite alternate;
    border-bottom: 1px solid rgba(15,23,42,.08);
  
    color: #0f172a;
    font-weight: 1000;}
  /* second subtle layer opposite direction */
  th::after{
    content:"";
    position:absolute;
    inset:0;
    background-image:
      linear-gradient(270deg,
        rgba(239,68,68,.18),
        rgba(59,130,246,.18),
        rgba(34,197,94,.18),
        rgba(250,204,21,.14),
        rgba(236,72,153,.14),
        rgba(45,212,191,.14),
        rgba(168,85,247,.14),
        rgba(239,68,68,.18)
      );
    background-size: 240% 100%;
    animation: thWave2 7s ease-in-out infinite alternate;
    mix-blend-mode:multiply;
    pointer-events:none;
  }
  @keyframes thWave{from{background-position:0% 50%}to{background-position:100% 50%}}
  @keyframes thWave2{from{background-position:100% 50%}to{background-position:0% 50%}}

</style>

<link rel="stylesheet" href="assets/sb_mobile.css?v=20260128">
<script src="assets/sb_mobile.js?v=20260128" defer></script>
</head>
<body>

<div class="topbar">
  <div class="top-left">
    <button class="menu-btn" type="button" onclick="document.body.classList.toggle('sidebar-open')">☰</button>
    <div class="pill"><span class="dot"></span> <?php echo htmlspecialchars($today_display); ?> · <?php echo htmlspecialchars($current_time); ?></div>
  </div>

  <!-- ✅ ONE LINE BIG -->
  <div class="brand"><span>Study Bridge</span></div>

  <div class="top-right">
    <div class="userline">
      <b><?php echo htmlspecialchars($displayUser); ?></b>
      <?php if($role!=='') echo ' <span style="opacity:.85">(' . htmlspecialchars($role) . ')</span>'; ?>
    </div>
</div>
</div>

<div class="layout">

  

  <main class="content">

    
<div class="formsbar">
<div class="forms-row">
    <a class="formbtn" href="attendance_in.php" title="Check In"><div class="fi"><img alt="CHECK IN" src="data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A//www.w3.org/2000/svg%22%20width%3D%2264%22%20height%3D%2264%22%20viewBox%3D%220%200%2064%2064%22%3E%0A%20%20%3Cdefs%3E%0A%20%20%20%20%3ClinearGradient%20id%3D%22g%22%20x1%3D%220%22%20x2%3D%221%22%3E%0A%20%20%20%20%20%20%3Cstop%20offset%3D%220%22%20stop-color%3D%22%2316a34a%22/%3E%0A%20%20%20%20%20%20%3Cstop%20offset%3D%221%22%20stop-color%3D%22%2322c55e%22/%3E%0A%20%20%20%20%3C/linearGradient%3E%0A%20%20%3C/defs%3E%0A%20%20%3Cpath%20d%3D%22M32%204c10.5%200%2019%208.5%2019%2019%200%2014.4-15.8%2033.9-18.1%2036.7a1.2%201.2%200%200%201-1.8%200C28.8%2056.9%2013%2037.4%2013%2023%2013%2012.5%2021.5%204%2032%204z%22%20fill%3D%22url%28%23g%29%22/%3E%0A%20%20%3Ccircle%20cx%3D%2232%22%20cy%3D%2223%22%20r%3D%229.5%22%20fill%3D%22%23ffffff%22%20opacity%3D%220.95%22/%3E%0A%20%20%3Cpath%20d%3D%22M27.2%2023.2l3.1%203.2%206.5-7%22%20fill%3D%22none%22%20stroke%3D%22%2316a34a%22%20stroke-width%3D%223.2%22%20stroke-linecap%3D%22round%22%20stroke-linejoin%3D%22round%22/%3E%0A%3C/svg%3E"></div><div class="ft">Check In</div></a>
    <a class="formbtn" href="attendance_out.php" title="Check Out"><div class="fi"><img alt="CHECK OUT" src="data:image/svg+xml;charset=utf-8,%3Csvg%20xmlns%3D%22http%3A//www.w3.org/2000/svg%22%20width%3D%2264%22%20height%3D%2264%22%20viewBox%3D%220%200%2064%2064%22%3E%0A%20%20%3Cdefs%3E%0A%20%20%20%20%3ClinearGradient%20id%3D%22r%22%20x1%3D%220%22%20x2%3D%221%22%3E%0A%20%20%20%20%20%20%3Cstop%20offset%3D%220%22%20stop-color%3D%22%23dc2626%22/%3E%0A%20%20%20%20%20%20%3Cstop%20offset%3D%221%22%20stop-color%3D%22%23ef4444%22/%3E%0A%20%20%20%20%3C/linearGradient%3E%0A%20%20%3C/defs%3E%0A%20%20%3Cpath%20d%3D%22M32%204c10.5%200%2019%208.5%2019%2019%200%2014.4-15.8%2033.9-18.1%2036.7a1.2%201.2%200%200%201-1.8%200C28.8%2056.9%2013%2037.4%2013%2023%2013%2012.5%2021.5%204%2032%204z%22%20fill%3D%22url%28%23r%29%22/%3E%0A%20%20%3Ccircle%20cx%3D%2232%22%20cy%3D%2223%22%20r%3D%229.5%22%20fill%3D%22%23ffffff%22%20opacity%3D%220.95%22/%3E%0A%20%20%3Cpath%20d%3D%22M28%2019l8%208M36%2019l-8%208%22%20fill%3D%22none%22%20stroke%3D%22%23dc2626%22%20stroke-width%3D%223.2%22%20stroke-linecap%3D%22round%22/%3E%0A%3C/svg%3E"></div><div class="ft">Check Out</div></a>
    <a class="formbtn" href="home_hours.php" title="Home Hours"><div class="fi">🏠</div><div class="ft">Home</div></a>
    <a class="formbtn" href="user_management.php" title="Users"><div class="fi">👥</div><div class="ft">Users</div></a>
    <a class="formbtn" href="forgot_password.php" title="Reset Password"><div class="fi">🔐</div><div class="ft">Reset</div></a>
    <a class="formbtn" href="attendance_list.php" title="Attendance List"><div class="fi">📋</div><div class="ft">List</div></a>
    <a class="formbtn" href="attendance_list.php" title="Edit Attendance (Last 12 Hours)"><div class="fi">✏️</div><div class="ft">Edit</div></a>
    <a class="formbtn" href="attendance_report.php" title="Reports"><div class="fi">📄</div><div class="ft">Reports</div></a>
<?php if (!empty($_SESSION['admin']) || !empty($_SESSION['admin_id'])) { ?>
    <a class="formbtn" href="backup.php" title="Database Backup"><div class="fi">🗄️</div><div class="ft">Backup</div></a>
<?php } ?>
  </div>
</div>


    <div class="card status-card">
      <div class="card-head">
        <div>Today’s Staff Status</div>
        <small><?php echo htmlspecialchars($today_display); ?></small>
      </div>

      <div class="table-container">
        <table>
          <thead>
            <tr>
              <th>Staff Name</th>
              <th>Designation</th>
              <th>Status</th>
              <th>In</th>
              <th>Out</th>
              <th>Home Hours</th>
              <th>Total Hours</th>
            </tr>
          </thead>
          <tbody>
            <?php
            $i = 0;
            if ($staff_q) {
              while ($s = mysqli_fetch_assoc($staff_q)) {
                $i++;
                $sid = (int)($s['id'] ?? 0);
                $att = $att_map[$sid] ?? null;

                $in = $att['in_time'] ?? '';
                $out = $att['out_time'] ?? '';
                $total = $att['total_hours'] ?? '';

                $badgeClass = 'b-absent';
                $statusText = 'Absent';

                if ($att) {
                  if (!empty($in) && empty($out)) {
                    $badgeClass = 'b-pending';
                    $statusText = 'In (Pending Out)';
                  } else {
                    $badgeClass = 'b-present';
                    $statusText = 'Present';
                  }
                }

                echo '<tr>';
                echo '<td>' . htmlspecialchars($s['name'] ?? '') . '</td>';
                echo '<td>' . htmlspecialchars($s['designation'] ?? '') . '</td>';
                echo '<td><span class="badge ' . $badgeClass . '">' . htmlspecialchars($statusText) . '</span></td>';
                echo '<td>' . htmlspecialchars(sb_fmt_time_12hr($in)) . '</td>';
                echo '<td>' . htmlspecialchars(sb_fmt_time_12hr($out)) . '</td>';
                $home = ($hasHomeHours && $att) ? ($att['home_hours'] ?? '') : '';
                echo '<td>' . htmlspecialchars((string)$home) . '</td>';
                echo '<td>' . htmlspecialchars((string)$total) . '</td>';
                echo '</tr>';
              }
            } else {
              echo '<tr><td colspan="7" style="padding:18px; text-align:center;">Staff list not found.</td></tr>';
            }
            ?>
          </tbody>
        </table>
      </div>
    </div>

    <div class="footer">
      <b>© 2026 Study Bridge</b>
    </div>

    <div class="devby">Developed by Atika</div>

  </main>
</div>


<a class="logout-fixed" href="login.php?logout=1" title="Logout">🚪 Logout</a>
</body>
</html>
