<?php
// home_hours_self.php (FIXED mysqlnd-free)
// 2 cards in one page: Home IN + Home OUT (editable time) + OUT comment required

require_once 'db.php';
require_once 'schema_guard.php';
require_once 'auth.php';
if (file_exists('ui.php')) { require_once 'ui.php'; }

if (function_exists('require_login')) { require_login(); }
$uid = (int)($_SESSION['user_id'] ?? 0);
if ($uid <= 0) { die("Unauthorized"); }

if (function_exists('require_permission')) { require_permission($conn, 'home_hours'); }

@date_default_timezone_set('Asia/Karachi');
function sb_fmt_ampm($timeStr){
  $timeStr = trim((string)$timeStr);
  if ($timeStr==='') return '';
  $ts = strtotime($timeStr);
  if ($ts===false) return $timeStr;
  return date('h:i A', $ts);
}

$today = date('Y-m-d');
$now_time_24 = date('H:i:s');
$now_time_ampm = date('h:i A');
if (function_exists('sb_ensure_attendance_schema')) {
  sb_ensure_attendance_schema($conn);
}

// Ensure extra home columns exist (safe)
if (!function_exists('sb_has_column')) {
  function sb_has_column(mysqli $conn, string $table, string $column): bool { return false; }
}
function sb_ensure_home_cols(mysqli $conn): void {
  $cols = [
    ['home_in_time',  'TIME NULL'],
    ['home_out_time', 'TIME NULL'],
    ['home_comment',  'TEXT NULL'],
    ['home_hours',    'DECIMAL(10,2) NULL'],
  ];
  foreach ($cols as $c) {
    [$name,$def] = $c;
    if (!sb_has_column($conn,'attendance',$name)) {
      @mysqli_query($conn, "ALTER TABLE attendance ADD COLUMN {$name} {$def}");
    }
  }
}
sb_ensure_home_cols($conn);

// Link staff_id
$staff_id = 0;
$staff_name = '';
$st = mysqli_prepare($conn, "SELECT id, name FROM staff_account WHERE user_id=? LIMIT 1");
if ($st) {
  mysqli_stmt_bind_param($st, "i", $uid);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st, $sid, $sname);
  if (mysqli_stmt_fetch($st)) { $staff_id = (int)$sid; $staff_name = (string)$sname; }
  mysqli_stmt_close($st);
}
if ($staff_id <= 0) {
  die("Your user is not linked with staff account (staff_account.user_id).");
}

// CSRF fallback
if (!function_exists('csrf_token')) {
  function csrf_token(): string {
    if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
    if (empty($_SESSION['_csrf'])) $_SESSION['_csrf'] = bin2hex(random_bytes(16));
    return $_SESSION['_csrf'];
  }
}
if (!function_exists('csrf_validate')) {
  function csrf_validate(string $t): bool {
    if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
    return isset($_SESSION['_csrf']) && hash_equals($_SESSION['_csrf'], (string)$t);
  }
}
$csrf = csrf_token();

function sb_hhmm_to_hhmmss(string $hhmm): string {
  $hhmm = trim($hhmm);
  if (!preg_match('/^([01]\d|2[0-3]):([0-5]\d)$/', $hhmm)) return '';
  return $hhmm . ':00';
}

function sb_get_row(mysqli $conn, int $staff_id, string $today): array {
  $row = ['id'=>0,'home_in_time'=>'','home_out_time'=>'','home_comment'=>'','home_hours'=>null];
  $q = mysqli_prepare($conn, "SELECT id, home_in_time, home_out_time, home_comment, home_hours
                             FROM attendance WHERE staff_id=? AND attendance_date=? LIMIT 1");
  if ($q) {
    mysqli_stmt_bind_param($q, "is", $staff_id, $today);
    mysqli_stmt_execute($q);
    mysqli_stmt_bind_result($q, $id, $hin, $hout, $hcom, $hhours);
    if (mysqli_stmt_fetch($q)) {
      $row['id']=(int)$id;
      $row['home_in_time']=(string)$hin;
      $row['home_out_time']=(string)$hout;
      $row['home_comment']=(string)$hcom;
      $row['home_hours']=$hhours;
    }
    mysqli_stmt_close($q);
  }
  return $row;
}

$msg=''; $type='warn';
$row = sb_get_row($conn, $staff_id, $today);

// Ensure attendance row exists helper
function sb_ensure_att_row(mysqli $conn, int $staff_id, int $uid, string $today): void {
  $q = mysqli_prepare($conn, "SELECT id FROM attendance WHERE staff_id=? AND attendance_date=? LIMIT 1");
  $id=0;
  if ($q) {
    mysqli_stmt_bind_param($q, "is", $staff_id, $today);
    mysqli_stmt_execute($q);
    mysqli_stmt_bind_result($q, $rid);
    if (mysqli_stmt_fetch($q)) $id=(int)$rid;
    mysqli_stmt_close($q);
  }
  if ($id<=0) {
    $ins = mysqli_prepare($conn, "INSERT INTO attendance (staff_id, attendance_date, user_id, created_by, updated_by, created_at, updated_at)
                                  VALUES (?,?,?,?,?,NOW(),NOW())");
    if ($ins) {
      mysqli_stmt_bind_param($ins, "isiii", $staff_id, $today, $uid, $uid, $uid);
      @mysqli_stmt_execute($ins);
      mysqli_stmt_close($ins);
    }
  }
}

if (isset($_POST['home_in'])) {
  if (!csrf_validate($_POST['csrf'] ?? '')) { $msg='CSRF error'; $type='err'; }
  else {
    $t = sb_hhmm_to_hhmmss((string)($_POST['home_in_hhmm'] ?? ''));
    if ($t==='') { $msg='Select valid Home IN time'; $type='warn'; }
    else {
      sb_ensure_att_row($conn, $staff_id, $uid, $today);
      $row = sb_get_row($conn, $staff_id, $today);
      if (!empty($row['home_in_time'])) { $msg='Home IN already marked'; $type='warn'; }
      else {
        $u = mysqli_prepare($conn, "UPDATE attendance SET home_in_time=?, updated_by=?, updated_at=NOW() WHERE staff_id=? AND attendance_date=? LIMIT 1");
        if ($u) {
          mysqli_stmt_bind_param($u, "siis", $t, $uid, $staff_id, $today);
          if (mysqli_stmt_execute($u)) { $msg='✅ Home IN saved'; $type='ok'; }
          else { $msg='Update failed: '.mysqli_stmt_error($u); $type='err'; }
          mysqli_stmt_close($u);
          $row = sb_get_row($conn, $staff_id, $today);
        } else { $msg='Server error'; $type='err'; }
      }
    }
  }
}

if (isset($_POST['home_out'])) {
  if (!csrf_validate($_POST['csrf'] ?? '')) { $msg='CSRF error'; $type='err'; }
  else {
    $t = sb_hhmm_to_hhmmss((string)($_POST['home_out_hhmm'] ?? ''));
    $comment = trim((string)($_POST['home_comment'] ?? ''));
    if ($t==='') { $msg='Select valid Home OUT time'; $type='warn'; }
    elseif ($comment==='') { $msg='Comment required on Home OUT'; $type='warn'; }
    else {
      $row = sb_get_row($conn, $staff_id, $today);
      if (empty($row['home_in_time'])) { $msg='Please Home IN first'; $type='warn'; }
      elseif (!empty($row['home_out_time'])) { $msg='Home OUT already marked'; $type='warn'; }
      else {
        $in_ts = strtotime($today.' '.$row['home_in_time']);
        $out_ts = strtotime($today.' '.$t);
        if ($in_ts===False or $out_ts===False or $out_ts <= $in_ts) { $msg='OUT must be greater than IN'; $type='err'; }
        else {
          $hours = round(($out_ts-$in_ts)/3600,2);
          $u = mysqli_prepare($conn, "UPDATE attendance SET home_out_time=?, home_hours=?, home_comment=?, updated_by=?, updated_at=NOW()
                                     WHERE staff_id=? AND attendance_date=? LIMIT 1");
          if ($u) {
            mysqli_stmt_bind_param($u, "sdsiss", $t, $hours, $comment, $uid, $staff_id, $today);
            if (mysqli_stmt_execute($u)) { $msg='✅ Home OUT saved'; $type='ok'; }
            else { $msg='Update failed: '.mysqli_stmt_error($u); $type='err'; }
            mysqli_stmt_close($u);
            $row = sb_get_row($conn, $staff_id, $today);
          } else { $msg='Server error'; $type='err'; }
        }
      }
    }
  }
}

?>
<!doctype html>
<html>
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>Home Hours</title>
<style>
:root{
  --ink:#0f172a;
  --muted:#475569;
  --stroke:rgba(15,23,42,.10);
  --shadow: 0 18px 40px rgba(2,6,23,.14);
  --r:22px;
}
body{
  font-family:system-ui,Arial;
  margin:0;
  color:var(--ink);
  background-color:#eef2f7;
  background-image:
    radial-gradient(1200px 520px at 15% 10%, rgba(59,130,246,.22), transparent 60%),
    radial-gradient(900px 420px at 85% 15%, rgba(148,163,184,.32), transparent 60%),
    radial-gradient(1100px 520px at 55% 95%, rgba(37,99,235,.18), transparent 65%),
    radial-gradient(760px 420px at 10% 90%, rgba(203,213,225,.45), transparent 60%);
}

.topbar{
  max-width:1120px;
  margin:14px auto 0;
  padding:0 12px;
  display:flex;
  align-items:center;
  justify-content:space-between;
  gap:12px;
}
.topbar .left{display:flex;align-items:center;gap:10px;flex-wrap:wrap}
.pill{
  padding:10px 14px;
  border-radius:999px;
  background:rgba(255,255,255,.72);
  border:1px solid var(--stroke);
  box-shadow: 0 10px 20px rgba(2,6,23,.08);
  backdrop-filter: blur(10px);
  font-weight:900;
}
.pill small{font-weight:800;color:var(--muted)}
a.btn{
  display:inline-flex;align-items:center;gap:10px;
  padding:11px 14px;border-radius:16px;
  background:rgba(255,255,255,.82);
  border:1px solid var(--stroke);
  text-decoration:none;color:var(--ink);
  font-weight:1000;
  box-shadow: 0 10px 20px rgba(2,6,23,.08);
  backdrop-filter: blur(8px);
}
a.btn:hover{filter:brightness(1.03)}

.wrap{
  max-width:1120px;
  margin:14px auto 18px;
  padding:0 12px;
  display:grid;
  grid-template-columns:1fr 1fr;
  gap:18px;
}
@media(max-width:980px){.wrap{grid-template-columns:1fr}}

.card{
  border-radius:var(--r);
  padding:24px;
  box-shadow: var(--shadow);
  border:1px solid rgba(255,255,255,.45);
  background: rgba(255,255,255,.78);
  backdrop-filter: blur(10px);
  min-height: 340px;
  position:relative;
  overflow:hidden;
}
.card:before{
  content:"";
  position:absolute; inset:-2px;
  background-image:
    radial-gradient(circle at 1px 1px, rgba(15,23,42,.07) 1px, transparent 1.8px),
    radial-gradient(700px 260px at 10% 0%, rgba(59,130,246,.20), transparent 60%),
    radial-gradient(680px 260px at 90% 10%, rgba(148,163,184,.25), transparent 60%),
    radial-gradient(650px 260px at 40% 100%, rgba(34,197,94,.14), transparent 60%);
  background-size: 18px 18px, auto, auto, auto;
  opacity:.85;
  pointer-events:none;
}
.card > *{position:relative; z-index:1;}

h3{margin:0 0 10px;font-size:20px;letter-spacing:.2px}
.meta{color:var(--muted);font-size:14px;margin:8px 0 12px;font-weight:800}

label{display:block;margin:0 0 6px;font-weight:1000;color:var(--ink);font-size:14px}
.row{display:grid;grid-template-columns: 1.25fr .75fr;gap:12px}
.row.single{grid-template-columns:1fr}
input[type="time"],input[type="text"],textarea{
  width:100%;
  padding:12px 14px;
  border-radius:16px;
  border:1px solid rgba(15,23,42,.12);
  background:rgba(255,255,255,.86);
  font-size:16px;
  box-shadow: 0 8px 16px rgba(2,6,23,.06) inset;
}
textarea{min-height:120px;resize:vertical}
.big{
  width:100%;
  padding:14px 18px;
  border-radius:18px;
  font-size:18px;
  border:0;
  cursor:pointer;
  font-weight:1100;
  box-shadow: 0 14px 28px rgba(2,6,23,.12);
}
.ok{background:rgba(34,197,94,.18)}
.danger{background:rgba(239,68,68,.16)}

.msg{max-width:1120px;margin:10px auto 0;padding:0 12px}
.msg .box{padding:12px 14px;border-radius:18px;background:rgba(255,255,255,.86);border:1px solid var(--stroke)}
.msg .ok{border-color:rgba(34,197,94,.45)}
.msg .warn{border-color:rgba(250,204,21,.55)}
.msg .err{border-color:rgba(239,68,68,.45);color:#8a1f1f}
</style>

<link rel="stylesheet" href="assets/sb_mobile.css?v=20260128">
<script src="assets/sb_mobile.js?v=20260128" defer></script>
</head>
<body>

<div class="topbar">
  <div class="left">
    <a class="btn" href="dashboard.php">🏠 Dashboard</a>
    <div class="pill">👤 <?php echo htmlspecialchars($staff_name ?? ''); ?></div>
  </div>
  <div class="right" style="display:flex;gap:10px;align-items:center;flex-wrap:wrap;justify-content:flex-end">
    <div class="pill"><small>Time:</small> <span id="liveClock"><?php echo htmlspecialchars($now_time_ampm ?? ''); ?></span></div>
    <a class="btn" href="logout.php">🚪 Logout</a>
  </div>
</div>
</div>

<?php if ($msg!==''){ ?>
  <div class="msg"><div class="box <?php echo htmlspecialchars($type); ?>"><?php echo htmlspecialchars($msg); ?></div></div>
<?php } ?>

<div class="wrap">
  <div class="card card-in">
    <h3>🏠 Home Hours — Check IN</h3>
    <div class="meta">Date: <b><?php echo htmlspecialchars($today); ?></b> | Staff: <b><?php echo htmlspecialchars($staff_name); ?></b></div>
    <form method="post" autocomplete="off">
      <input type="hidden" name="csrf" value="<?php echo htmlspecialchars($csrf); ?>">
      <div class="row">
        <div>
          <label>Home IN Time (Editable)</label>
          <input type="time" name="home_in_hhmm" required value="<?php echo htmlspecialchars(!empty($row['home_in_time']) ? substr($row['home_in_time'],0,5) : substr($now_time_24,0,5)); ?>">
        </div>
        <div>
          <label>Status</label>
          <input type="text" readonly value="<?php echo htmlspecialchars(!empty($row['home_in_time']) ? ('Home IN: '.sb_fmt_ampm($row['home_in_time'])) : 'Not IN'); ?>">
        </div>
      </div>
      <div style="margin-top:12px">
        <button class="big ok" type="submit" name="home_in">🟢 Home Check IN</button>
      </div>
    </form>
  </div>

  <div class="card card-out">
    <h3>🏠 Home Hours — Check OUT</h3>
    <div class="meta">Date: <b><?php echo htmlspecialchars($today); ?></b> | Now: <b><?php echo htmlspecialchars(substr($now_time_24,0,5)); ?></b></div>
    <form method="post" autocomplete="off">
      <input type="hidden" name="csrf" value="<?php echo htmlspecialchars($csrf); ?>">
      <div class="row">
        <div>
          <label>Home OUT Time (Editable)</label>
          <input type="time" name="home_out_hhmm" required value="<?php echo htmlspecialchars(!empty($row['home_out_time']) ? substr($row['home_out_time'],0,5) : substr($now_time_24,0,5)); ?>">
        </div>
        <div>
          <label>Home IN Time</label>
          <input type="text" readonly value="<?php echo htmlspecialchars(!empty($row['home_in_time']) ? sb_fmt_ampm($row['home_in_time']) : 'IN first'); ?>">
        </div>
      </div>

      <div class="row single" style="margin-top:10px">
        <div>
          <label>Comment / Work (Required)</label>
          <textarea name="home_comment" rows="4" required><?php echo htmlspecialchars((string)($_POST['home_comment'] ?? ($row['home_comment'] ?? ''))); ?></textarea>
        </div>
      </div>

      <div style="margin-top:12px">
        <button class="big danger" type="submit" name="home_out">🔴 Home Check OUT</button>
      </div>

      <div class="meta" style="margin-top:10px">
        Home OUT: <b><?php echo htmlspecialchars(!empty($row['home_out_time']) ? sb_fmt_ampm($row['home_out_time']) : '-'); ?></b>
        | Home Hours: <b><?php echo htmlspecialchars(($row['home_hours']!==null && $row['home_hours']!=='') ? (string)$row['home_hours'] : '-'); ?></b>
      </div>
    </form>
  </div>
</div>


<script>
(function(){
  function pad(n){ return (n<10?'0':'')+n; }
  function to12h(h,m){
    var ampm = h>=12 ? 'PM':'AM';
    var hh = h%12; if(hh===0) hh=12;
    return pad(hh)+':'+pad(m)+' '+ampm;
  }
  function tick(){
    var d = new Date();
    var el = document.getElementById('liveClock');
    if(el){ el.textContent = to12h(d.getHours(), d.getMinutes()); }
  }
  tick();
  setInterval(tick, 1000*20);
})();
</script>

</body>
</html>
