<?php
require_once 'db.php';
require_once 'auth.php';
require_once 'ui.php';

require_login();
require_permission($conn, 'home_hours');

@date_default_timezone_set('Asia/Karachi');


// ------------------------------
// Schema helpers (avoid silent failures)
// ------------------------------
function sb_table_exists($conn, $table) {
  $sql = "SHOW TABLES LIKE ?";
  $st = mysqli_prepare($conn, $sql);
  if (!$st) return false;
  mysqli_stmt_bind_param($st, 's', $table);
  mysqli_stmt_execute($st);
  mysqli_stmt_store_result($st);
  $ok = (mysqli_stmt_num_rows($st) > 0);
  mysqli_stmt_close($st);
  return $ok;
}

function sb_table_has_columns($conn, $table, $cols) {
  $cols = (array)$cols;
  $found = [];
  $q = mysqli_prepare($conn, "SHOW COLUMNS FROM `{$table}`");
  if (!$q) return false;
  mysqli_stmt_execute($q);
  $rows = stmt_fetch_all_assoc($q);
  mysqli_stmt_close($q);
  foreach ($rows as $r) {
    $found[strtolower($r['Field'])] = true;
  }
  foreach ($cols as $c) {
    if (empty($found[strtolower($c)])) return false;
  }
  return true;
}


function sb_column_type($conn, $table, $column) {
  $sql = "SHOW FULL COLUMNS FROM `$table` LIKE ?";
  $st = mysqli_prepare($conn, $sql);
  if (!$st) return null;
  mysqli_stmt_bind_param($st, 's', $column);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st, $Field, $Type, $Collation, $Null, $Key, $Default, $Extra, $Privileges, $Comment);
  $type = null;
  if (mysqli_stmt_fetch($st)) {
    $type = $Type; // e.g. "year(4)" or "date"
  }
  mysqli_stmt_close($st);
  return $type;
}

function sb_stmt_error_text($stmt, $conn) {
  $s = '';
  if ($stmt) $s .= 'stmt_error=' . mysqli_stmt_error($stmt) . '; ';
  if ($conn) $s .= 'mysqli_error=' . mysqli_error($conn) . '; ';
  return trim($s);
}

if (!function_exists('stmt_fetch_all_assoc')) {
  /**
   * Fetch all rows from a prepared statement as associative arrays WITHOUT mysqlnd.
   */
  function stmt_fetch_all_assoc($stmt) {
    $rows = [];
    if (!$stmt) return $rows;

    // Buffer result to allow metadata + multiple fetches reliably
    @mysqli_stmt_store_result($stmt);

    $meta = mysqli_stmt_result_metadata($stmt);
    if (!$meta) return $rows;

    $fields = mysqli_fetch_fields($meta);
    $row = [];
    $bind = [];
    foreach ($fields as $field) {
      $row[$field->name] = null;
      $bind[] = &$row[$field->name];
    }

    // Bind by reference
    call_user_func_array('mysqli_stmt_bind_result', array_merge([$stmt], $bind));

    while (mysqli_stmt_fetch($stmt)) {
      // Copy the current row (because $row is reused by reference)
      $rows[] = array_merge([], $row);
    }

    mysqli_free_result($meta);
    return $rows;
  }

  /**
   * Fetch the first row from a prepared statement as an associative array WITHOUT mysqlnd.
   */
  function stmt_fetch_one_assoc($stmt) {
    $all = stmt_fetch_all_assoc($stmt);
    return isset($all[0]) ? $all[0] : null;
  }
}

$uid = current_user_id();
$work_date_type = sb_column_type($conn, 'home_work', 'work_date');
$now = date('H:i:s');
$work_date_value = null;
$work_date_is_numeric = false;
// If DB column is YEAR/INT, store only year (integer). If DATE, store full Y-m-d (string).
if ($work_date_type && (stripos($work_date_type, 'year') !== false || stripos($work_date_type, 'int') !== false)) {
  $work_date_value = (int)date('Y');
  $work_date_is_numeric = true;
} else {
  $work_date_value = date('Y-m-d');
  $work_date_is_numeric = false;
}
$work_date_label = ($work_date_is_numeric ? (string)$work_date_value : date('Y', strtotime((string)$work_date_value)));

// Linked staff
$staff_id = 0;
$staff_name = '';
$st = mysqli_prepare($conn, "SELECT id, name FROM staff_account WHERE user_id=? LIMIT 1");
if ($st) {
  mysqli_stmt_bind_param($st, 'i', $uid);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st, $sid, $sname);
  if (mysqli_stmt_fetch($st)) {
    $staff_id = (int)$sid;
    $staff_name = (string)$sname;
  }
  mysqli_stmt_close($st);
}


// Admin can choose any staff if not linked
$admin_pick_staff = false;
if ($staff_id <= 0 && is_admin()) {
  $admin_pick_staff = true;
  $picked = (int)($_POST['staff_id'] ?? ($_SESSION['pick_staff_id'] ?? 0));
  if ($picked > 0) {
    $ps = mysqli_prepare($conn, "SELECT id, name FROM staff_account WHERE id=? LIMIT 1");
    if ($ps) {
      mysqli_stmt_bind_param($ps, "i", $picked);
      mysqli_stmt_execute($ps);
      mysqli_stmt_bind_result($ps, $pid, $pname);
      if (mysqli_stmt_fetch($ps)) {
        $staff_id = (int)$pid;
        $staff_name = (string)$pname;
        $_SESSION['pick_staff_id'] = $staff_id;
      }
      mysqli_stmt_close($ps);
    }
  }
}

if ($staff_id <= 0 && !is_admin()) {
  sb_page_header('Home Hours');
  echo '<div class="msg err">Your user is not linked with staff account. Admin please set staff_account.user_id = users.id.</div>';
  sb_page_footer();
  exit;
}

// Ensure table exists (safe)
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS home_work (
  id INT AUTO_INCREMENT PRIMARY KEY,
  staff_id INT NOT NULL,
  user_id INT NOT NULL,
  work_date DATE NOT NULL,
  start_time TIME NOT NULL,
  end_time TIME DEFAULT NULL,
  note TEXT DEFAULT NULL,
  created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
  KEY idx_hw_user_date (user_id, work_date),
  KEY idx_hw_staff_date (staff_id, work_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4" );

// If table already existed with different schema, show a clear message instead of failing silently.
if (sb_table_exists($conn, 'home_work') && !sb_table_has_columns($conn, 'home_work', ['staff_id','user_id','work_date','start_time','end_time','note'])) {
  sb_page_header('Home Hours');
  echo '<div class="msg err">Database table <b>home_work</b> exists but schema does not match required columns (staff_id,user_id,work_date,start_time,end_time,note). Please update the table schema.</div>';
  if (is_admin()) {
    echo '<div class="muted" style="margin-top:8px">Tip: run ALTER TABLE to add missing columns or rename the old table.</div>';
  }
  sb_page_footer();
  exit;
}


$msg = '';
$type = 'warn';
$csrf = csrf_token();

// Find open session for today
$open = null;
$q = mysqli_prepare($conn, "SELECT id, start_time FROM home_work WHERE user_id=? AND work_date=? AND end_time IS NULL ORDER BY id DESC LIMIT 1");
if ($q) {
  if ($work_date_is_numeric) {
    mysqli_stmt_bind_param($q, 'ii', $uid, $work_date_value);
  } else {
    mysqli_stmt_bind_param($q, 'is', $uid, $work_date_value);
  }
  mysqli_stmt_execute($q);
  $open = stmt_fetch_one_assoc($q);
  mysqli_stmt_close($q);
}

if (isset($_POST['start'])) {
  if (!csrf_validate($_POST['csrf'] ?? '')) {
    $msg = 'Security error (CSRF).';
    $type = 'err';
  } elseif ($open) {
    $msg = 'You already started home work. Please end it first.';
    $type = 'warn';
  } else {
    $ins = mysqli_prepare($conn, "INSERT INTO home_work (staff_id, user_id, work_date, start_time) VALUES (?,?,?,?)");
    if ($ins) {
      if ($work_date_is_numeric) {
        mysqli_stmt_bind_param($ins, 'iiis', $staff_id, $uid, $work_date_value, $now);
      } else {
        mysqli_stmt_bind_param($ins, 'iiss', $staff_id, $uid, $work_date_value, $now);
      }
      if (mysqli_stmt_execute($ins)) {
        $msg = '✅ Home work started.';
        $type = 'ok';
        // refresh open
        $open = ['id'=>mysqli_insert_id($conn), 'start_time'=>$now];
      } else {
        $msg = '❌ Start failed.';
        if (is_admin()) { $msg .= ' ' . sb_stmt_error_text($ins, $conn); }
        $type = 'err';
      }
      mysqli_stmt_close($ins);
    } else {
      $msg = 'Server error.';
      $type = 'err';
    }
  }
}

if (isset($_POST['end'])) {
  if (!csrf_validate($_POST['csrf'] ?? '')) {
    $msg = 'Security error (CSRF).';
    $type = 'err';
  } elseif (!$open) {
    $msg = 'No active home work found. Please start first.';
    $type = 'warn';
  } else {
    $note = trim((string)($_POST['note'] ?? ''));
    if ($note === '') {
      $msg = 'Please write work/comment before ending.';
      $type = 'warn';
    } else {
      $up = mysqli_prepare($conn, "UPDATE home_work SET end_time=?, note=?, updated_at=NOW() WHERE id=? AND user_id=? LIMIT 1");
      if ($up) {
        $id = (int)$open['id'];
        mysqli_stmt_bind_param($up, 'ssii', $now, $note, $id, $uid);
        if (mysqli_stmt_execute($up)) {
          $msg = '✅ Home work ended.';
          $type = 'ok';
          $open = null;
        } else {
          $msg = '❌ End failed.';
          if (is_admin()) { $msg .= ' ' . sb_stmt_error_text($up, $conn); }
          $type = 'err';
        }
        mysqli_stmt_close($up);
      } else {
        $msg = 'Server error.';
        $type = 'err';
      }
    }
  }
}

// Recent rows
$rows = [];
$r = mysqli_prepare($conn, "SELECT work_date, start_time, end_time, note FROM home_work WHERE user_id=? ORDER BY id DESC LIMIT 20");
if ($r) {
  mysqli_stmt_bind_param($r, 'i', $uid);
  mysqli_stmt_execute($r);
  $rows = stmt_fetch_all_assoc($r);
  mysqli_stmt_close($r);
}

sb_page_header('Home Hours');

echo '<div class="muted">Date: <b>' . htmlspecialchars($work_date_label) . '</b> &nbsp; Time: <b>' . htmlspecialchars($now) . '</b></div>';
if ($staff_name !== '') echo '<div class="muted" style="margin-top:6px">Staff: <b>' . htmlspecialchars($staff_name) . '</b></div>';

if ($msg !== '') echo '<div class="msg ' . htmlspecialchars($type) . '">' . htmlspecialchars($msg) . '</div>';
?>

<form method="post" autocomplete="off">
  <input type="hidden" name="csrf" value="<?php echo htmlspecialchars($csrf); ?>">

<?php if (!empty($admin_pick_staff)) { ?>
  <label>Select Staff (Admin)</label>
  <select name="staff_id" required>
    <option value="">-- Select Staff --</option>
    <?php
      $rs = mysqli_query($conn, "SELECT id, name FROM staff_account ORDER BY name");
      $cur = (int)$staff_id;
      if ($rs) {
        while ($r = mysqli_fetch_assoc($rs)) {
          $id = (int)$r['id'];
          $nm = (string)$r['name'];
          $sel = ($id === $cur) ? 'selected' : '';
          echo '<option value="'.htmlspecialchars((string)$id).'" '.$sel.'>'.htmlspecialchars($nm).' (ID '.$id.')</option>';
        }
        mysqli_free_result($rs);
      }
    ?>
  </select>
  <div class="muted" style="margin-top:6px">Admin: select staff, then Start/End Home Work.</div>
  <div style="height:10px"></div>
<?php } ?>


  <?php if (!$open) { ?>
    <div class="actions">
      <button class="btn primary" type="submit" name="start">🏠 Start Home Work</button>
    </div>
  <?php } else { ?>
    <div class="msg warn">Active: started at <b><?php echo htmlspecialchars((string)$open['start_time']); ?></b></div>
    <label>Work / Comment (Report)</label>
    <textarea name="note" placeholder="Ghar me aaj kya kaam kiya (2-3 lines)" required><?php echo htmlspecialchars((string)($_POST['note'] ?? '')); ?></textarea>
    <div class="actions">
      <button class="btn danger" type="submit" name="end">✅ End Home Work</button>
    </div>
  <?php } ?>
</form>

<hr style="border:none;border-top:1px solid var(--stroke);margin:16px 0">

<div class="h1" style="font-size:15px">Recent Home Hours</div>
<style>
  table{width:100%;border-collapse:separate;border-spacing:0;margin-top:8px;}
  th,td{padding:10px;border-bottom:1px solid var(--stroke);text-align:left;font-size:13.5px;vertical-align:top;}
  th{font-weight:1000;color:var(--muted);background:rgba(37,99,235,.06);} 
</style>
<table>
  <thead>
    <tr><th>Date</th><th>Start</th><th>End</th><th>Comment</th></tr>
  </thead>
  <tbody>
    <?php foreach ($rows as $rw): ?>
      <tr>
        <td><?php echo htmlspecialchars($rw['work_date']); ?></td>
        <td><?php echo htmlspecialchars((string)$rw['start_time']); ?></td>
        <td><?php echo htmlspecialchars((string)($rw['end_time'] ?? '')); ?></td>
        <td><?php echo nl2br(htmlspecialchars((string)($rw['note'] ?? ''))); ?></td>
      </tr>
    <?php endforeach; ?>
  </tbody>
</table>

<?php sb_page_footer();
