<?php
// schema_guard_fixed.php
// ✅ mysqlnd-free: no mysqli_stmt_get_result() usage
// Safe for LiteSpeed / shared hosting

if (!defined('SB_GUARD_LOADED')) {
    define('SB_GUARD_LOADED', true);
}

/** Check if a table exists in current DB */
function sb_table_exists(mysqli $conn, string $table): bool
{
    $sql = "SELECT COUNT(*) FROM information_schema.TABLES
            WHERE TABLE_SCHEMA = DATABASE()
              AND TABLE_NAME = ?";

    $stmt = $conn->prepare($sql);
    if (!$stmt) return false;

    $stmt->bind_param("s", $table);
    $stmt->execute();
    $stmt->bind_result($count);
    $stmt->fetch();
    $stmt->close();

    return ($count > 0);
}

/** Check if a column exists in a table (mysqlnd-free) */
function sb_has_column(mysqli $conn, string $table, string $column): bool
{
    $sql = "SELECT COUNT(*) FROM information_schema.COLUMNS
            WHERE TABLE_SCHEMA = DATABASE()
              AND TABLE_NAME = ?
              AND COLUMN_NAME = ?";

    $stmt = $conn->prepare($sql);
    if (!$stmt) return false;

    $stmt->bind_param("ss", $table, $column);
    $stmt->execute();
    $stmt->bind_result($count);
    $stmt->fetch();
    $stmt->close();

    return ($count > 0);
}

/** Ensure user_permissions table exists (dashboard.php uses this) */
function sb_ensure_permissions_schema(mysqli $conn): void
{
    if (!sb_table_exists($conn, 'user_permissions')) {
        $sql = "CREATE TABLE IF NOT EXISTS user_permissions (
            user_id INT NOT NULL PRIMARY KEY,
            staff_manage TINYINT(1) NOT NULL DEFAULT 0,
            attendance_in TINYINT(1) NOT NULL DEFAULT 1,
            attendance_out TINYINT(1) NOT NULL DEFAULT 1,
            home_hours TINYINT(1) NOT NULL DEFAULT 1,
            reports TINYINT(1) NOT NULL DEFAULT 0,
            updated_at TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        @mysqli_query($conn, $sql);
    }
}

/** Ensure attendance columns exist for OUT comment / auditing */
function sb_ensure_attendance_schema(mysqli $conn): void
{
    // OUT comment columns (keep both names for compatibility)
    if (!sb_has_column($conn, 'attendance', 'out_note')) {
        @mysqli_query($conn, "ALTER TABLE attendance ADD COLUMN out_note TEXT NULL");
    }
    if (!sb_has_column($conn, 'attendance', 'out_work')) {
        @mysqli_query($conn, "ALTER TABLE attendance ADD COLUMN out_work TEXT NULL");
    }

    // Audit columns
    if (!sb_has_column($conn, 'attendance', 'created_by')) {
        @mysqli_query($conn, "ALTER TABLE attendance ADD COLUMN created_by INT NULL");
    }
    if (!sb_has_column($conn, 'attendance', 'updated_by')) {
        @mysqli_query($conn, "ALTER TABLE attendance ADD COLUMN updated_by INT NULL");
    }
    if (!sb_has_column($conn, 'attendance', 'created_at')) {
        @mysqli_query($conn, "ALTER TABLE attendance ADD COLUMN created_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP");
    }
    if (!sb_has_column($conn, 'attendance', 'updated_at')) {
        @mysqli_query($conn, "ALTER TABLE attendance ADD COLUMN updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP");
    }
}
