<?php
require_once 'db.php';
require_once 'auth.php';
require_once 'delete_log.php';
require_once 'ui.php';

require_login();

@date_default_timezone_set('Asia/Karachi');

$uid   = current_user_id();
$admin = is_admin();
$csrf  = csrf_token();

/**
 * Helpers
 */
function sb_has_column(mysqli $conn, string $table, string $column): bool {
  $sql = "SELECT COUNT(*) AS c
          FROM information_schema.COLUMNS
          WHERE TABLE_SCHEMA = DATABASE()
            AND TABLE_NAME = ?
            AND COLUMN_NAME = ?
          LIMIT 1";
  $st = mysqli_prepare($conn, $sql);
  if (!$st) return false;
  mysqli_stmt_bind_param($st, 'ss', $table, $column);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st, $c);
  $ok = false;
  if (mysqli_stmt_fetch($st)) $ok = ((int)$c > 0);
  mysqli_stmt_close($st);
  return $ok;
}

function sb_bind_params(mysqli_stmt $st, string $types, array &$params): bool {
  if ($types === '' || empty($params)) return true;
  $bind = [];
  $bind[] = $st;
  $bind[] = $types;
  foreach ($params as $k => $v) {
    $bind[] = &$params[$k];
  }
  return (bool)call_user_func_array('mysqli_stmt_bind_param', $bind);
}

function sb_stmt_prepare(mysqli $conn, string $sql, string $types = '', array $params = [], ?mysqli_stmt &$out = null): bool {
  $st = mysqli_prepare($conn, $sql);
  if (!$st) return false;
  if ($types !== '') {
    if (!sb_bind_params($st, $types, $params)) {
      mysqli_stmt_close($st);
      return false;
    }
  }
  $out = $st;
  return true;
}

function sb_stmt_fetch_one(mysqli_stmt $st): ?array {
  // Works with and without mysqlnd
  if (function_exists('mysqli_stmt_get_result')) {
    $res = @mysqli_stmt_get_result($st);
    if ($res) {
      $row = mysqli_fetch_assoc($res);
      return $row ? $row : null;
    }
  }

  $meta = mysqli_stmt_result_metadata($st);
  if (!$meta) return null;

  $fields = mysqli_fetch_fields($meta);
  if (!$fields) return null;

  $row = [];
  $bind = [];
  foreach ($fields as $f) {
    $row[$f->name] = null;
    $bind[] = &$row[$f->name];
  }

  $args = array_merge([$st], $bind);
  call_user_func_array('mysqli_stmt_bind_result', $args);

  if (mysqli_stmt_fetch($st)) return $row;
  return null;
}

function sb_get_atika_id(mysqli $conn): int {
  $atika = 'atika';
  $st = mysqli_prepare($conn, "SELECT id FROM users WHERE LOWER(username)=LOWER(?) LIMIT 1");
  if (!$st) return 0;
  mysqli_stmt_bind_param($st, 's', $atika);
  mysqli_stmt_execute($st);
  mysqli_stmt_bind_result($st, $id);
  $rid = 0;
  if (mysqli_stmt_fetch($st)) $rid = (int)$id;
  mysqli_stmt_close($st);
  return $rid;
}

function sb_is_protected_user(int $id, string $username, int $atika_id): bool {
  if ($atika_id > 0 && $id === $atika_id) return true;
  return (strtolower(trim($username)) === 'atika');
}

/**
 * Column detection (avoids "prepare failed" if your DB doesn't have these yet)
 */
$HAS_ROLE       = sb_has_column($conn, 'users', 'role');
$HAS_STATUS     = sb_has_column($conn, 'users', 'status');
$HAS_DESIGN     = sb_has_column($conn, 'users', 'designation');
$HAS_CREATED_AT = sb_has_column($conn, 'users', 'created_at');
$ATIKA_ID       = sb_get_atika_id($conn);

/**
 * Staff mode: can edit only own username, and can view users list (no edits of others)
 */
if (!$admin) {
  $msg = '';
  $err = '';

  if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_me'])) {
    if (!csrf_validate($_POST['csrf'] ?? '')) {
      $err = 'Security error (CSRF). Please refresh.';
    } else {
      $new_username = trim((string)($_POST['username'] ?? ''));
      if ($new_username === '' || !preg_match('/^[A-Za-z0-9_.-]{2,30}$/', $new_username)) {
        $err = 'Invalid username. Use 2-30 chars: letters, numbers, underscore, dot or dash.';
      } else {
        // unique check
        $st = mysqli_prepare($conn, "SELECT id FROM users WHERE username=? AND id<>? LIMIT 1");
        if ($st) {
          mysqli_stmt_bind_param($st, "si", $new_username, $uid);
          mysqli_stmt_execute($st);
          mysqli_stmt_bind_result($st, $ex_id);
          $exists = false;
          if (mysqli_stmt_fetch($st)) $exists = true;
          mysqli_stmt_close($st);
          if ($exists) $err = 'This username is already taken.';
        }

        if ($err === '') {
          $up = mysqli_prepare($conn, "UPDATE users SET username=? WHERE id=? LIMIT 1");
          if ($up) {
            mysqli_stmt_bind_param($up, "si", $new_username, $uid);
            if (mysqli_stmt_execute($up)) {
              $_SESSION['username'] = $new_username;
              $msg = '✅ Your username has been updated.';
            } else {
              $err = 'Update failed: ' . htmlspecialchars(mysqli_stmt_error($up));
            }
            mysqli_stmt_close($up);
          } else {
            $err = 'Server error (prepare failed): ' . htmlspecialchars(mysqli_error($conn));
          }
        }
      }
    }
  }

  // Fetch my user (safe even if optional columns are missing)
  $me = ['id' => (int)$uid, 'username' => (string)current_username(), 'role' => (string)current_role()];
  $cols = ['id', 'username'];
  if ($HAS_ROLE)   $cols[] = 'role';
  if ($HAS_DESIGN) $cols[] = 'designation';
  $sql = "SELECT " . implode(',', $cols) . " FROM users WHERE id=? LIMIT 1";
  $st = mysqli_prepare($conn, $sql);
  if ($st) {
    mysqli_stmt_bind_param($st, "i", $uid);
    mysqli_stmt_execute($st);

    // mysqlnd-free bind
    $row = null;
    if (function_exists('mysqli_stmt_get_result')) {
      $row = sb_stmt_fetch_one($st);
    } else {
      // fallback: bind dynamically (most setups have mysqlnd; if not, keep defaults)
    }
    if ($row) {
      $me['id'] = (int)$row['id'];
      $me['username'] = (string)$row['username'];
      if ($HAS_ROLE && isset($row['role'])) $me['role'] = (string)$row['role'];
    }
    mysqli_stmt_close($st);
  }

  // Users list (view-only)
  $users = [];
  $cols = ['id', 'username'];
  if ($HAS_ROLE) $cols[] = 'role';
  $q = mysqli_query($conn, "SELECT " . implode(',', $cols) . " FROM users ORDER BY id ASC");
  while ($q && ($r = mysqli_fetch_assoc($q))) $users[] = $r;

  sb_page_header('Users');
  ?>
  <style>
    .card{max-width:980px;margin:16px auto;background:#fff;border-radius:16px;padding:16px;border:1px solid rgba(15,23,42,.12);box-shadow:0 18px 44px rgba(2,6,23,.10)}
    table{width:100%;border-collapse:collapse}
    th,td{padding:10px;border-bottom:1px solid rgba(15,23,42,.10);text-align:left}
    .msg{padding:10px 12px;border-radius:12px;margin:10px 0}
    .ok{background:#dcfce7;border:1px solid #86efac}
    .err{background:#fee2e2;border:1px solid #fecaca}
    .muted{color:#475569}
    input{padding:10px 12px;border-radius:12px;border:1px solid rgba(15,23,42,.18);width:100%}
    .btn{padding:10px 14px;border-radius:12px;border:0;cursor:pointer}
    .primary{background:#2563eb;color:#fff}
  </style>

  <div class="card">
    <h2 style="margin:0 0 6px">My Profile</h2>
    <div class="muted">You can update only your own username. You cannot edit other users.</div>

    <?php if ($msg): ?><div class="msg ok"><?php echo htmlspecialchars($msg); ?></div><?php endif; ?>
    <?php if ($err): ?><div class="msg err"><?php echo htmlspecialchars($err); ?></div><?php endif; ?>

    <form method="post" style="margin-top:12px">
      <input type="hidden" name="csrf" value="<?php echo htmlspecialchars($csrf); ?>">
      <label class="muted">Username</label>
      <input name="username" value="<?php echo htmlspecialchars((string)$me['username']); ?>" required>
      <button class="btn primary" type="submit" name="update_me" style="margin-top:10px">Save</button>
    </form>
  </div>

  <div class="card">
    <h2 style="margin:0 0 10px">Users List (view only)</h2>
    <table>
      <thead>
        <tr><th>ID</th><th>Username</th><?php if ($HAS_ROLE): ?><th>Role</th><?php endif; ?></tr>
      </thead>
      <tbody>
      <?php foreach ($users as $u): ?>
        <tr>
          <td><?php echo (int)$u['id']; ?></td>
          <td><?php echo htmlspecialchars((string)$u['username']); ?></td>
          <?php if ($HAS_ROLE): ?><td><?php echo htmlspecialchars((string)$u['role']); ?></td><?php endif; ?>
        </tr>
      <?php endforeach; ?>
      </tbody>
    </table>
    <div class="muted" style="margin-top:10px">Note: Salary and other users' reports are not accessible in staff mode.</div>
  </div>
  <?php
  sb_page_footer();
  exit;
}

/**
 * ADMIN MODE
 * Requirement: Admin "Atika" must NEVER be deleted (and should remain active admin).
 */

// Ensure salary tables exist (safe no-op if already there)
@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS staff_account (
  id INT AUTO_INCREMENT PRIMARY KEY,
  user_id INT NOT NULL UNIQUE,
  name VARCHAR(190) NOT NULL,
  salary DECIMAL(12,2) NOT NULL DEFAULT 0
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

@mysqli_query($conn, "CREATE TABLE IF NOT EXISTS salary_history (
  id INT AUTO_INCREMENT PRIMARY KEY,
  staff_id INT NOT NULL,
  salary DECIMAL(12,2) NOT NULL,
  changed_by INT NULL,
  changed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");

function sb_admin_count(mysqli $conn): int {
  $c = 0;
  $q = mysqli_query($conn, "SELECT COUNT(*) AS c FROM users WHERE role='admin'");
  if ($q && ($r = mysqli_fetch_assoc($q))) $c = (int)($r['c'] ?? 0);
  return $c;
}

$success = '';
$error   = '';
$edit    = null;

// DELETE (admin only) + audit log
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['do_delete'])) {
  if (!csrf_validate($_POST['csrf'] ?? '')) {
    $error = 'Security error (CSRF).';
  } else {
    $del_id = (int)($_POST['id'] ?? 0);
    if ($del_id <= 0) {
      $error = 'Invalid user id.';
    } else {
      // Snapshot (only select columns that exist)
      $cols = ['id', 'username'];
      if ($HAS_ROLE)       $cols[] = 'role';
      if ($HAS_STATUS)     $cols[] = 'status';
      if ($HAS_DESIGN)     $cols[] = 'designation';
      if ($HAS_CREATED_AT) $cols[] = 'created_at';
      $sql = "SELECT " . implode(',', $cols) . " FROM users WHERE id=? LIMIT 1";

      $g = mysqli_prepare($conn, $sql);
      $snap = null;
      if ($g) {
        mysqli_stmt_bind_param($g, 'i', $del_id);
        mysqli_stmt_execute($g);
        $snap = sb_stmt_fetch_one($g);
        mysqli_stmt_close($g);
      }

      if (!$snap) {
        $error = 'User not found.';
      } else {
        $snap_user = (string)($snap['username'] ?? '');
        $snap_role = (string)($snap['role'] ?? '');

        // ✅ Never delete admin Atika
        if (sb_is_protected_user((int)$snap['id'], $snap_user, $ATIKA_ID)) {
          $error = '❌ Admin "Atika" cannot be deleted in any case.';
        } else {
          // Don't allow deleting the last ACTIVE admin (safety)
          if ($snap_role === 'admin') {
            $this_is_active_admin = true;
            if ($HAS_STATUS) $this_is_active_admin = (((int)($snap['status'] ?? 1)) === 1);

            $q = "SELECT COUNT(*) AS c FROM users WHERE role='admin' AND id<>?";
            if ($HAS_STATUS) $q .= " AND status=1";
            $stn = mysqli_prepare($conn, $q);
            $other_active_admins = 0;
            if ($stn) {
              mysqli_stmt_bind_param($stn, 'i', $del_id);
              mysqli_stmt_execute($stn);
              mysqli_stmt_bind_result($stn, $other_active_admins);
              mysqli_stmt_fetch($stn);
              mysqli_stmt_close($stn);
            }

            if ($this_is_active_admin && (int)$other_active_admins <= 0) {
              $error = '❌ You cannot delete the last active admin user.';
            }
          }

          if ($error === '') {
            // Log then delete
            sb_log_delete($conn, 'users', $del_id, current_user_id(), $snap, 'Admin delete');

            $st = mysqli_prepare($conn, 'DELETE FROM users WHERE id=? LIMIT 1');
            if ($st) {
              mysqli_stmt_bind_param($st, 'i', $del_id);
              if (mysqli_stmt_execute($st)) {
                // Also delete linked staff_account row (prevents ghost rows on dashboard)
                $ds = mysqli_prepare($conn, 'DELETE FROM staff_account WHERE user_id=? LIMIT 1');
                if ($ds) { mysqli_stmt_bind_param($ds, 'i', $del_id); @mysqli_stmt_execute($ds); mysqli_stmt_close($ds); }
                $success = '✅ User deleted (logged in deleted_records).';
              } else {
                $error = '❌ Delete failed: ' . htmlspecialchars(mysqli_stmt_error($st));
              }
              mysqli_stmt_close($st);
            } else {
              $error = 'Server error (prepare failed): ' . htmlspecialchars(mysqli_error($conn));
            }
          }
        }
      }
    }
  }
}

// EDIT LOAD
if (isset($_GET['edit'])) {
  $eid = (int)$_GET['edit'];
  $cols = ['id', 'username'];
  if ($HAS_ROLE)   $cols[] = 'role';
  if ($HAS_STATUS) $cols[] = 'status';
  if ($HAS_DESIGN) $cols[] = 'designation';
  $sql = "SELECT " . implode(',', $cols) . " FROM users WHERE id=? LIMIT 1";
  $st = mysqli_prepare($conn, $sql);
  if ($st) {
    mysqli_stmt_bind_param($st, 'i', $eid);
    mysqli_stmt_execute($st);
    $edit = sb_stmt_fetch_one($st);
    mysqli_stmt_close($st);
  }
  if (!$edit) $edit = null;
}

// ADD / UPDATE
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['save_user'])) {
  if (!csrf_validate($_POST['csrf'] ?? '')) {
    $error = 'Security error (CSRF).';
  } else {
    $id       = (int)($_POST['id'] ?? 0);
    $username = trim((string)($_POST['username'] ?? ''));
    $role_req = (string)($_POST['role'] ?? 'staff');
    $role     = ($role_req === 'admin') ? 'admin' : 'staff';
    $status   = ((int)($_POST['status'] ?? 1) === 1) ? 1 : 0;

    $salary   = trim((string)($_POST['salary'] ?? ''));
    if ($salary === '') $salary = '0';
    $salary_val = (float)$salary;

    if ($username === '' || !preg_match('/^[A-Za-z0-9_.-]{2,30}$/', $username)) {
      $error = 'Username required (2-30 chars: letters, numbers, underscore, dot, dash).';
    } else {
      // Identify if this record is Atika (protected)
      $is_atika = false;
      if ($id > 0) {
        $is_atika = sb_is_protected_user($id, $username, $ATIKA_ID);
      } else {
        $is_atika = (strtolower($username) === 'atika');
      }

      // ✅ Enforce: Atika must stay admin + active
      if ($is_atika) {
        $role = 'admin';
        $status = 1;
      }


      // ✅ Protect Atika from username changes (keeps protection stable)
      // ✅ Also prevent removing/disabling the last active admin (safety)
      if ($error === '' && $id > 0) {
        // Fetch current record (mysqlnd-free)
        $cols = ['id','username'];
        if ($HAS_ROLE)   $cols[] = 'role';
        if ($HAS_STATUS) $cols[] = 'status';
        $sqlc = "SELECT " . implode(',', $cols) . " FROM users WHERE id=? LIMIT 1";
        $stc = mysqli_prepare($conn, $sqlc);
        $cur = null;
        if ($stc) {
          mysqli_stmt_bind_param($stc, 'i', $id);
          mysqli_stmt_execute($stc);
          $cur = sb_stmt_fetch_one($stc);
          mysqli_stmt_close($stc);
        }

        if ($cur) {
          $cur_username = (string)($cur['username'] ?? '');
          $cur_role     = (string)($cur['role'] ?? '');
          $cur_status   = (int)($cur['status'] ?? 1);

          // If this is Atika by ID, username must not change (prevents losing protection)
          if ($ATIKA_ID > 0 && $id === $ATIKA_ID) {
            if (strtolower($username) !== strtolower($cur_username)) {
              $error = '❌ Admin "Atika" username cannot be changed.';
              $username = $cur_username;
            }
          }

          // Safety: don't allow removing/demoting/disabling the last ACTIVE admin
          if ($error === '' && $HAS_ROLE && $cur_role === 'admin') {
            $this_is_active_admin = true;
            if ($HAS_STATUS) $this_is_active_admin = ($cur_status === 1);

            // Will this user remain an active admin after save?
            $will_be_active_admin = ($role === 'admin');
            if ($HAS_STATUS) $will_be_active_admin = $will_be_active_admin && ($status === 1);

            if ($this_is_active_admin && !$will_be_active_admin) {
              $q = "SELECT COUNT(*) AS c FROM users WHERE role='admin' AND id<>?";
              if ($HAS_STATUS) $q .= " AND status=1";
              $stn = mysqli_prepare($conn, $q);
              $other_active_admins = 0;
              if ($stn) {
                mysqli_stmt_bind_param($stn, 'i', $id);
                mysqli_stmt_execute($stn);
                mysqli_stmt_bind_result($stn, $other_active_admins);
                mysqli_stmt_fetch($stn);
                mysqli_stmt_close($stn);
              }
              if ((int)$other_active_admins <= 0) {
                $error = '❌ You cannot remove/disable the last active admin.';
              }
            }
          }
        }
      }

      // Unique username check (ignore current id)
      if ($error === '') {
        $chk = mysqli_prepare($conn, "SELECT id FROM users WHERE username=? AND id<>? LIMIT 1");
        if ($chk) {
          mysqli_stmt_bind_param($chk, "si", $username, $id);
          mysqli_stmt_execute($chk);
          mysqli_stmt_bind_result($chk, $exid);
          if (mysqli_stmt_fetch($chk)) $error = 'This username is already taken.';
          mysqli_stmt_close($chk);
        }
      }

      if ($error === '') {
        $designation = '';
        if ($HAS_DESIGN) $designation = trim((string)($_POST['designation'] ?? ''));

        if ($id > 0) {
          // UPDATE
          $sets = [];
          $types = '';
          $params = [];

          $sets[] = "username=?";
          $types .= 's';
          $params[] = $username;

          if ($HAS_ROLE) {
            $sets[] = "role=?";
            $types .= 's';
            $params[] = $role;
          }
          if ($HAS_STATUS) {
            $sets[] = "status=?";
            $types .= 'i';
            $params[] = $status;
          }
          if ($HAS_DESIGN) {
            $sets[] = "designation=?";
            $types .= 's';
            $params[] = $designation;
          }

          $types .= 'i';
          $params[] = $id;

          $sql = "UPDATE users SET " . implode(',', $sets) . " WHERE id=? LIMIT 1";
          $st = mysqli_prepare($conn, $sql);
          if (!$st) {
            $error = 'Server error (prepare failed): ' . htmlspecialchars(mysqli_error($conn));
          } else {
            if (!sb_bind_params($st, $types, $params)) {
              $error = 'Server error (bind failed).';
            } elseif (mysqli_stmt_execute($st)) {
              $success = '✅ User updated.';
            } else {
              $error = '❌ Update failed: ' . htmlspecialchars(mysqli_stmt_error($st));
            }
            mysqli_stmt_close($st);
          }

          // Optional password change
          $newpw = trim((string)($_POST['new_password'] ?? ''));
          if ($error === '' && $newpw !== '') {
            $hash = password_hash($newpw, PASSWORD_DEFAULT);
            $pwst = mysqli_prepare($conn, 'UPDATE users SET password=? WHERE id=? LIMIT 1');
            if ($pwst) {
              mysqli_stmt_bind_param($pwst, 'si', $hash, $id);
              @mysqli_stmt_execute($pwst);
              mysqli_stmt_close($pwst);
              $success = '✅ User updated (password changed).';
            }
          }

          // Sync salary for staff users
          if ($error === '' && $role === 'staff') {
            $sid = 0; $oldsal = null;
            $gs = mysqli_prepare($conn, "SELECT id, salary FROM staff_account WHERE user_id=? LIMIT 1");
            if ($gs) {
              mysqli_stmt_bind_param($gs,'i',$id);
              mysqli_stmt_execute($gs);
              mysqli_stmt_bind_result($gs,$sid2,$sal2);
              if (mysqli_stmt_fetch($gs)) { $sid=(int)$sid2; $oldsal=(float)$sal2; }
              mysqli_stmt_close($gs);
            }

            if ($sid <= 0) {
              $insS = mysqli_prepare($conn, "INSERT INTO staff_account (user_id, name, salary) VALUES (?,?,?)");
              if ($insS) {
                $nm = $username;
                mysqli_stmt_bind_param($insS,'isd',$id,$nm,$salary_val);
                @mysqli_stmt_execute($insS);
                mysqli_stmt_close($insS);
                $sid = (int)mysqli_insert_id($conn);
                $oldsal = null;
              }
            } else {
              $upS = mysqli_prepare($conn, "UPDATE staff_account SET salary=?, name=? WHERE id=? LIMIT 1");
              if ($upS) {
                $nm = $username;
                mysqli_stmt_bind_param($upS,'dsi',$salary_val,$nm,$sid);
                @mysqli_stmt_execute($upS);
                mysqli_stmt_close($upS);
              }
            }

            // log salary change
            if ($sid > 0 && ($oldsal === null || abs($oldsal - $salary_val) > 0.009)) {
              $hs = mysqli_prepare($conn, "INSERT INTO salary_history (staff_id, salary, changed_by) VALUES (?,?,?)");
              if ($hs) {
                $by = current_user_id();
                mysqli_stmt_bind_param($hs,'idi',$sid,$salary_val,$by);
                @mysqli_stmt_execute($hs);
                mysqli_stmt_close($hs);
              }
            }
          }

        } else {
          // INSERT (new user)
          $pw = trim((string)($_POST['password'] ?? ''));
          if ($pw === '') {
            $error = 'Password required for new user.';
          } else {
            $hash = password_hash($pw, PASSWORD_DEFAULT);

            $cols = ['username', 'password'];
            $ph   = ['?', '?'];
            $types = 'ss';
            $params = [$username, $hash];

            if ($HAS_ROLE) {
              $cols[] = 'role';
              $ph[] = '?';
              $types .= 's';
              $params[] = $role;
            }
            if ($HAS_STATUS) {
              $cols[] = 'status';
              $ph[] = '?';
              $types .= 'i';
              $params[] = $status;
            }
            if ($HAS_DESIGN) {
              $cols[] = 'designation';
              $ph[] = '?';
              $types .= 's';
              $params[] = $designation;
            }

            $sql = "INSERT INTO users (" . implode(',', $cols) . ") VALUES (" . implode(',', $ph) . ")";
            $st = mysqli_prepare($conn, $sql);
            if (!$st) {
              $error = 'Server error (prepare failed): ' . htmlspecialchars(mysqli_error($conn));
            } else {
              if (!sb_bind_params($st, $types, $params)) {
                $error = 'Server error (bind failed).';
              } elseif (mysqli_stmt_execute($st)) {
                $success = '✅ User added.';
                $new_id = (int)mysqli_insert_id($conn);

                if ($new_id > 0 && $role === 'staff') {
                  $insS = mysqli_prepare($conn, "INSERT INTO staff_account (user_id, name, salary) VALUES (?,?,?)");
                  if ($insS) {
                    $nm = $username;
                    mysqli_stmt_bind_param($insS,'isd',$new_id,$nm,$salary_val);
                    @mysqli_stmt_execute($insS);
                    mysqli_stmt_close($insS);
                    $sid = (int)mysqli_insert_id($conn);
                    if ($sid > 0) {
                      $hs = mysqli_prepare($conn, "INSERT INTO salary_history (staff_id, salary, changed_by) VALUES (?,?,?)");
                      if ($hs) {
                        $by = current_user_id();
                        mysqli_stmt_bind_param($hs,'idi',$sid,$salary_val,$by);
                        @mysqli_stmt_execute($hs);
                        mysqli_stmt_close($hs);
                      }
                    }
                  }
                }
              } else {
                $error = '❌ Insert failed: ' . htmlspecialchars(mysqli_stmt_error($st));
              }
              mysqli_stmt_close($st);
            }
          }
        }
      }
    }
  }
}

// LIST users
$users = [];
$cols = ['id', 'username'];
if ($HAS_ROLE)   $cols[] = 'role';
if ($HAS_STATUS) $cols[] = 'status';
$sql = "SELECT " . implode(',', $cols) . " FROM users ORDER BY id DESC";
$q = mysqli_query($conn, $sql);
while ($q && ($r = mysqli_fetch_assoc($q))) $users[] = $r;

sb_page_header('User Management');
?>

<style>
  table{width:100%;border-collapse:separate;border-spacing:0;margin-top:12px;}
  th,td{padding:10px;border-bottom:1px solid var(--stroke);text-align:left;font-size:13.5px;}
  th{font-weight:1000;color:var(--muted);background:rgba(37,99,235,.06);}
  tr:hover td{background:rgba(2,6,23,.02);}
  .grid{display:grid;grid-template-columns:1fr 1fr;gap:12px;}
  @media(max-width:720px){.grid{grid-template-columns:1fr;}}
  .note{color:var(--muted);font-size:12px;margin-top:6px}
</style>

<?php
if ($success !== '') echo '<div class="msg ok">'.htmlspecialchars($success).'</div>';
if ($error   !== '') echo '<div class="msg err">'.htmlspecialchars($error).'</div>';
?>

<form method="post" autocomplete="off">
  <input type="hidden" name="csrf" value="<?php echo htmlspecialchars($csrf); ?>">
  <input type="hidden" name="id" value="<?php echo (int)($edit['id'] ?? 0); ?>">

  <div class="grid">
    <div>
      <label>Username</label>
      <input name="username" required value="<?php echo htmlspecialchars((string)($edit['username'] ?? ($_POST['username'] ?? ''))); ?>">
      <div class="note">2-30 chars: letters, numbers, underscore, dot, dash</div>
    </div>

    <div>
      <label>Role</label>
      <?php $r = (string)($edit['role'] ?? ($_POST['role'] ?? 'staff')); ?>
      <select name="role" <?php echo (sb_is_protected_user((int)($edit['id'] ?? 0), (string)($edit['username'] ?? ''), $ATIKA_ID) ? 'disabled' : ''); ?>>
        <option value="staff" <?php echo ($r==='staff'?'selected':''); ?>>Staff</option>
        <option value="admin" <?php echo ($r==='admin'?'selected':''); ?>>Admin</option>
      </select>
      <?php if (sb_is_protected_user((int)($edit['id'] ?? 0), (string)($edit['username'] ?? ''), $ATIKA_ID)): ?>
        <div class="note">Admin Atika is protected and always stays Admin.</div>
      <?php endif; ?>
    </div>

    <div>
      <label>Salary (optional)</label>
      <?php
        $salv = '';
        $eid_for_salary = (int)($edit['id'] ?? 0);
        if ($eid_for_salary > 0) {
          $sal = 0;
          $gs = mysqli_prepare($conn, 'SELECT salary FROM staff_account WHERE user_id=? LIMIT 1');
          if ($gs) {
            mysqli_stmt_bind_param($gs,'i',$eid_for_salary);
            mysqli_stmt_execute($gs);
            mysqli_stmt_bind_result($gs,$sal);
            if (mysqli_stmt_fetch($gs)) { $salv = (string)$sal; }
            mysqli_stmt_close($gs);
          }
        }
        $salary_val_show = (string)($_POST['salary'] ?? $salv);
      ?>
      <input name="salary" type="number" step="0.01" value="<?php echo htmlspecialchars($salary_val_show); ?>">
    </div>

    <div>
      <label>Status</label>
      <?php $stt = (int)($edit['status'] ?? ($_POST['status'] ?? 1)); ?>
      <select name="status" <?php echo (sb_is_protected_user((int)($edit['id'] ?? 0), (string)($edit['username'] ?? ''), $ATIKA_ID) ? 'disabled' : ''); ?>>
        <option value="1" <?php echo ($stt===1?'selected':''); ?>>Active</option>
        <option value="0" <?php echo ($stt===0?'selected':''); ?>>Disabled</option>
      </select>
      <?php if (!$HAS_STATUS): ?>
        <div class="note">Note: Your users table has no <b>status</b> column. Status selection is ignored.</div>
      <?php endif; ?>
    </div>

    <div>
      <label><?php echo ($edit ? 'New Password (optional)' : 'Password'); ?></label>
      <input type="password" name="<?php echo ($edit ? 'new_password' : 'password'); ?>" <?php echo ($edit ? '' : 'required'); ?>>
    </div>

    <div>
      <label>Designation (optional)</label>
      <input name="designation" value="<?php echo htmlspecialchars((string)($edit['designation'] ?? ($_POST['designation'] ?? ''))); ?>" <?php echo ($HAS_DESIGN ? '' : 'disabled'); ?>>
      <?php if (!$HAS_DESIGN): ?>
        <div class="note">Note: Your users table has no <b>designation</b> column. This field is ignored.</div>
      <?php endif; ?>
    </div>
  </div>

  <div class="actions">
    <button class="btn primary" type="submit" name="save_user">💾 Save</button>
    <?php if ($edit) { ?><a class="btn" href="user_management.php">➕ Add New</a><?php } ?>
  </div>
</form>

<table>
  <thead>
    <tr>
      <th>ID</th>
      <th>Username</th>
      <?php if ($HAS_ROLE): ?><th>Role</th><?php endif; ?>
      <?php if ($HAS_STATUS): ?><th>Status</th><?php endif; ?>
      <th style="width:220px">Actions</th>
    </tr>
  </thead>
  <tbody>
    <?php foreach ($users as $u): ?>
      <?php
        $pid = (int)$u['id'];
        $pun = (string)$u['username'];
        $is_protected = sb_is_protected_user($pid, $pun, $ATIKA_ID);
      ?>
      <tr>
        <td><?php echo $pid; ?></td>
        <td>
          <b><?php echo htmlspecialchars($pun); ?></b>
          <?php if ($is_protected): ?>
            <span class="pill" style="margin-left:6px;background:#dcfce7;border:1px solid #86efac;color:#166534;">PROTECTED</span>
          <?php endif; ?>
        </td>
        <?php if ($HAS_ROLE): ?><td><?php echo htmlspecialchars((string)$u['role']); ?></td><?php endif; ?>
        <?php if ($HAS_STATUS): ?><td><?php echo ((int)($u['status'] ?? 1)===1 ? 'Active' : 'Disabled'); ?></td><?php endif; ?>
        <td>
          <a class="btn" href="user_management.php?edit=<?php echo $pid; ?>">Edit</a>

          <?php if ($is_protected): ?>
            <button class="btn danger" type="button" disabled title="Admin Atika cannot be deleted">Delete</button>
          <?php else: ?>
            <form method="post" style="display:inline" onsubmit="return confirm('Delete user? Admin only.');">
              <input type="hidden" name="csrf" value="<?php echo htmlspecialchars($csrf); ?>">
              <input type="hidden" name="id" value="<?php echo $pid; ?>">
              <button class="btn danger" type="submit" name="do_delete">Delete</button>
            </form>
          <?php endif; ?>
        </td>
      </tr>
    <?php endforeach; ?>
  </tbody>
</table>

<?php sb_page_footer();