<?php
include 'config/config.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// If already logged in
if (!empty($_SESSION['user'])) {
    header("Location: dashboard.php");
    exit();
}

$error = "";

/* ---------------------------
   Simple rate limit (per session)
---------------------------- */
$maxAttempts   = 10;
$windowSeconds = 15 * 60; // 15 minutes

if (!isset($_SESSION['login_rate'])) {
    $_SESSION['login_rate'] = ['count' => 0, 'start' => time()];
}
$rate = &$_SESSION['login_rate'];

// reset window
if ((time() - (int)$rate['start']) > $windowSeconds) {
    $rate = ['count' => 0, 'start' => time()];
}

$blocked = ((int)$rate['count'] >= $maxAttempts);
if ($blocked) {
    $error = "Too many attempts. Try again after 15 minutes.";
}

/* ---------------------------
   Handle login
---------------------------- */
if (isset($_POST['login']) && !$blocked) {

    $username = trim($_POST['username'] ?? '');
    $password = (string)($_POST['password'] ?? '');

    if ($username === '' || $password === '') {
        $error = "Username and password are required.";
    } else {

        // Prefer password_hash (Roadmap), fallback to legacy password if needed.
        $stmt = $conn->prepare("
            SELECT id, username, role,
                   COALESCE(NULLIF(password_hash,''), NULLIF(password,'')) AS pass_any
            FROM users
            WHERE username = ?
            LIMIT 1
        ");

        if (!$stmt) {
            error_log("LOGIN PREPARE ERROR: " . $conn->error);
            $error = "Server error. Please try again.";
        } else {

            $stmt->bind_param("s", $username);

            if (!$stmt->execute()) {
                error_log("LOGIN EXECUTE ERROR: " . $stmt->error);
                $error = "Server error. Please try again.";
                $stmt->close();
            } else {

                $stmt->store_result();

                if ($stmt->num_rows <= 0) {
                    $rate['count']++;
                    $error = "User Not Found!";
                    $stmt->close();
                } else {

                    $id = 0;
                    $dbUsername = '';
                    $role = '';
                    $stored = '';

                    $stmt->bind_result($id, $dbUsername, $role, $stored);
                    $stmt->fetch();
                    $stmt->close();

                    $stored = (string)$stored;

                    $placeholders = ['PASTE_NEW_HASH_HERE', 'PASTE_REAL_HASH_HERE', 'PASTE_COPIED_HASH_HERE'];
                    if (in_array($stored, $placeholders, true) || $stored === '') {
                        $rate['count']++;
                        $error = "Password hash missing/placeholder. Please set a real password hash in DB.";
                    } else {

                        $looksHashed =
                            (strpos($stored, '$2y$') === 0) ||
                            (strpos($stored, '$2a$') === 0) ||
                            (strpos($stored, '$2b$') === 0) ||
                            (strpos($stored, '$argon2') === 0);

                        $ok = $looksHashed ? password_verify($password, $stored) : hash_equals($stored, $password);

                        if ($ok) {

                            // Auto-upgrade into password_hash if needed
                            if (!$looksHashed || password_needs_rehash($stored, PASSWORD_DEFAULT)) {
                                $newHash = password_hash($password, PASSWORD_DEFAULT);

                                $upd = $conn->prepare("UPDATE users SET password_hash = ? WHERE id = ?");
                                if ($upd) {
                                    $uid = (int)$id;
                                    $upd->bind_param("si", $newHash, $uid);
                                    $upd->execute();
                                    $upd->close();
                                }
                            }

                            session_regenerate_id(true);

                            $_SESSION['user'] = $dbUsername;
                            $_SESSION['role'] = $role;
                            $_SESSION['user_id'] = (int)$id;

                            $_SESSION['login_rate'] = ['count' => 0, 'start' => time()];

                            header("Location: dashboard.php");
                            exit();

                        } else {
                            $rate['count']++;
                            $error = "Invalid Password!";
                        }
                    }
                }
            }
        }
    }
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>Login - The Foundation Montessori</title>
    <link rel="stylesheet" href="assets/css/style.css?v=<?php echo APP_VERSION; ?>">
</head>
<body>
<div class="login-box">
    <h2>The Foundation Montessori</h2>

    <form method="POST" autocomplete="off">
        <input type="text" name="username" placeholder="Username" required><br>
        <input type="password" name="password" placeholder="Password" required><br>
        <button type="submit" name="login">Login</button>
    </form>

    <?php if (!empty($error)) echo "<p style='color:red;'>" . htmlspecialchars($error, ENT_QUOTES, 'UTF-8') . "</p>"; ?>
</div>
</body>
</html>
