<?php
include '../config/config.php';
if (session_status() === PHP_SESSION_NONE) { session_start(); }
if (!isset($_SESSION['user'])) {
    header("Location: ../index.php");
    exit();
}

$error = "";

/** Generate next Admission No like TFM-STD-0001 (uses MAX(id)) */
function generate_admission_no(mysqli $conn): string {
    $nextId = 1;
    $q = $conn->query("SELECT MAX(id) AS mx FROM students");
    if ($q) {
        $row = $q->fetch_assoc();
        if (!empty($row['mx'])) $nextId = (int)$row['mx'] + 1;
    }
    return 'TFM-STD-' . str_pad((string)$nextId, 4, '0', STR_PAD_LEFT);
}

function fetch_all(mysqli $conn, string $sql): array {
    $rows = [];
    if ($res = $conn->query($sql)) {
        while($r = $res->fetch_assoc()) $rows[] = $r;
        $res->free();
    }
    return $rows;
}

$classes  = fetch_all($conn, "SELECT id, name FROM classes ORDER BY name ASC");
$sessions = fetch_all($conn, "SELECT id, title, is_active FROM sessions ORDER BY is_active DESC, id DESC");
$active_session_id = null;
foreach ($sessions as $s) { if ((int)$s['is_active'] === 1) { $active_session_id = (int)$s['id']; break; } }
if ($active_session_id === null && !empty($sessions)) $active_session_id = (int)$sessions[0]['id'];

$class_id = (int)($_POST['class_id'] ?? ($_GET['class_id'] ?? 0));
$sections = [];
if ($class_id > 0) {
    $st = $conn->prepare("SELECT id, name FROM sections WHERE class_id=? ORDER BY name ASC");
    if ($st) {
        $st->bind_param("i", $class_id);
        $st->execute();
        $sections = stmt_fetch_all_assoc($st);
        $st->close();
    }
}

// Handle submit
if (isset($_POST['submit'])) {

    $admission_no = trim($_POST['admission_no'] ?? '');
    if ($admission_no === '') $admission_no = generate_admission_no($conn);

    $first_name = trim($_POST['first_name'] ?? '');
    $last_name  = trim($_POST['last_name'] ?? '');
    $name       = trim($first_name . ' ' . $last_name);

    $gender     = trim($_POST['gender'] ?? '');
    $dob        = $_POST['dob'] ?? null;

    // Montessori (minimal guardian info here)
    $father_name = trim($_POST['father_name'] ?? '');
    $guardian_phone = trim($_POST['guardian_phone'] ?? '');

    $class_id   = (int)($_POST['class_id'] ?? 0);
    $section_id = (int)($_POST['section_id'] ?? 0);
    $session_id = (int)($_POST['session_id'] ?? 0);
    $issue_date = $_POST['issue_date'] ?? date('Y-m-d');

    if ($first_name === '' || $last_name === '' || $gender === '' || $father_name === '' || $class_id<=0 || $section_id<=0 || $session_id<=0) {
        $error = "Please fill all required fields.";
    } else {

        // Photo upload (optional)
        $photoPath = null;
        if (!empty($_FILES['photo']['name'])) {
            $targetDir = __DIR__ . '/../uploads/students/';
            if (!is_dir($targetDir)) { @mkdir($targetDir, 0755, true); }
            $ext = strtolower(pathinfo($_FILES['photo']['name'], PATHINFO_EXTENSION));
            if (!in_array($ext, ['jpg','jpeg','png','webp','gif'])) {
                $error = "Invalid photo type. Please upload jpg/png/webp.";
            } else {
                $safeName = preg_replace('/[^a-zA-Z0-9_\-]/', '_', $admission_no);
                $fileName = $safeName . '_' . time() . '.' . $ext;
                $dest = $targetDir . $fileName;
                if (move_uploaded_file($_FILES['photo']['tmp_name'], $dest)) {
                    $photoPath = 'uploads/students/' . $fileName;
                }
            }
        }

        if ($error === "") {

            // Fetch class/section/session titles for legacy text columns (keeps old screens safe)
            $class_txt = '';
            $section_txt = '';
            $session_txt = '';

            $q = $conn->prepare("SELECT name FROM classes WHERE id=?");
            if ($q) { $q->bind_param("i", $class_id); $q->execute(); $q->bind_result($class_txt); $q->fetch(); $q->close(); }

            $q = $conn->prepare("SELECT name FROM sections WHERE id=?");
            if ($q) { $q->bind_param("i", $section_id); $q->execute(); $q->bind_result($section_txt); $q->fetch(); $q->close(); }

            $q = $conn->prepare("SELECT title FROM sessions WHERE id=?");
            if ($q) { $q->bind_param("i", $session_id); $q->execute(); $q->bind_result($session_txt); $q->fetch(); $q->close(); }

            // Insert student (Roadmap columns + keep legacy columns if they exist)
            $sql = "INSERT INTO students
                (admission_no, first_name, last_name, name, gender, dob,
                 class_id, section_id, session_id, issue_date, photo,
                 student_id, father_name, phone, class, section, session)
                VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";

            $stmt = $conn->prepare($sql);
            if (!$stmt) {
                $error = "DB error: " . $conn->error . " — Please run db_upgrade_v6.sql in phpMyAdmin.";
            } else {
                // student_id legacy mirrors admission_no
                $student_id_legacy = $admission_no;

                $stmt->bind_param(
                    "ssssssiiissssssss",
                    $admission_no, $first_name, $last_name, $name, $gender, $dob,
                    $class_id, $section_id, $session_id, $issue_date, $photoPath,
                    $student_id_legacy, $father_name, $guardian_phone,
                    $class_txt, $section_txt, $session_txt
                );

                if ($stmt->execute()) {
                    $newStudentId = (int)$stmt->insert_id;
                    $stmt->close();

                    // Insert/Update guardian row
                    $g = $conn->prepare("INSERT INTO guardians (student_id, father_name, phone, relation)
                                         VALUES (?,?,?,?)
                                         ON DUPLICATE KEY UPDATE father_name=VALUES(father_name), phone=VALUES(phone), relation=VALUES(relation)");
                    if ($g) {
                        $rel = "Father";
                        $g->bind_param("isss", $newStudentId, $father_name, $guardian_phone, $rel);
                        $g->execute();
                        $g->close();
                    }

                    header("Location: student_list.php");
                    exit();
                } else {
                    $error = "Save failed: " . $stmt->error;
                    $stmt->close();
                }
            }
        }
    }
}

$generated_id = generate_admission_no($conn);
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Add Student</title>
  <link rel="stylesheet" href="../assets/css/style.css?v=<?php echo APP_VERSION; ?>">
</head>
<body>

<div class="sidebar">
  <div class="brand">
    <div class="brand-logo"><img src="../assets/img/logo.svg" alt="Logo"></div>
    <div class=\"brand-text\">
      <div class=\"brand-title\">The Foundation Montessori</div>
      <div class=\"brand-sub\">School System</div>
    </div>
  </div>

  <div class="nav">
    <a class="nav-link" href="../dashboard.php">Dashboard</a>
    <a class="nav-link active" href="../students/add_student.php">Students</a>
    <a class="nav-link" href="../attendance/attendance.php">Attendance</a>
    <a class="nav-link" href="../staff/staff_list.php">Staff</a>
    <a class="nav-link" href="../fees/fees.php">Fees</a>
    <a class="nav-link" href="../exams/exams.php">Exams</a>
    <a class="nav-link" href="../reports/reports.php">Reports</a>
  </div>
</div>

<div class="content">
  <div class="topbar" style="min-height:115px;">
    <div style="display:flex;gap:10px;align-items:center;">
      <a class="btn btn-white" href="../dashboard.php">Dashboard</a>
      <a class="btn btn-white" href="javascript:history.back()">Back</a>
    </div>

    <div style="text-align:center;flex:1;">
      <div class="topbar-title" style="font-size:18px;">The Foundation Montessori</div>
      <div style="margin-top:6px;font-weight:700;"><?php echo htmlspecialchars($_SESSION['user'] ?? 'User'); ?></div>
    </div>

    <div style="display:flex;align-items:center;gap:10px;">
      <a class="btn btn-red" href="../logout.php">Log out</a>
    </div>
  </div>

  <div style="height:14px"></div>

  <div class="card">
    <h2 class="card-title">Add Student</h2>

    <?php if(!empty($error)){ echo '<div class="alert">'.htmlspecialchars($error).'</div>'; } ?>

    <form method="POST" enctype="multipart/form-data">
      <div class="grid-2">
        <div>
          <label>Admission No (Auto)</label>
          <input type="text" name="admission_no" value="<?php echo htmlspecialchars($generated_id); ?>" readonly>
        </div>
        <div>
          <label>Issue Date</label>
          <input type="date" name="issue_date" value="<?php echo date('Y-m-d'); ?>" required>
        </div>
      </div>

      <div class="grid-2">
        <div>
          <label>First Name</label>
          <input type="text" name="first_name" required>
        </div>
        <div>
          <label>Last Name</label>
          <input type="text" name="last_name" required>
        </div>
      </div>

      <div class="grid-2">
        <div>
          <label>Gender</label>
          <select name="gender" required>
            <option value="">Select</option>
            <option>Male</option>
            <option>Female</option>
            <option>Other</option>
          </select>
        </div>
        <div>
          <label>Date of Birth</label>
          <input type="date" name="dob">
        </div>
      </div>

      <div class="grid-2">
        <div>
          <label>Father / Guardian Name</label>
          <input type="text" name="father_name" required>
        </div>
        <div>
          <label>Guardian Phone</label>
          <input type="text" name="guardian_phone" placeholder="03xx...">
        </div>
      </div>

      <div class="grid-2">
        <div>
          <label>Profile Picture Upload</label>
          <input type="file" name="photo" accept="image/*">
        </div>
        <div></div>
      </div>

      <div class="grid-3">
        <div>
          <label>Class</label>
          <select name="class_id" required onchange="window.location='?class_id='+this.value;">
            <option value="">Select</option>
            <?php foreach($classes as $c): ?>
              <option value="<?php echo (int)$c['id']; ?>" <?php echo ($class_id==(int)$c['id'])?'selected':''; ?>>
                <?php echo htmlspecialchars($c['name']); ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>
        <div>
          <label>Section</label>
          <select name="section_id" required>
            <option value="">Select</option>
            <?php foreach($sections as $sc): ?>
              <option value="<?php echo (int)$sc['id']; ?>"><?php echo htmlspecialchars($sc['name']); ?></option>
            <?php endforeach; ?>
          </select>
          <div style="font-size:12px;color:#6b7280;margin-top:6px;">If section list is empty, select class again.</div>
        </div>
        <div>
          <label>Session</label>
          <select name="session_id" required>
            <option value="">Select</option>
            <?php foreach($sessions as $s): ?>
              <option value="<?php echo (int)$s['id']; ?>" <?php echo ((int)$s['id']===(int)$active_session_id)?'selected':''; ?>>
                <?php echo htmlspecialchars($s['title']); ?><?php echo ((int)$s['is_active']===1)?' (Active)':''; ?>
              </option>
            <?php endforeach; ?>
          </select>
        </div>
      </div>

      <div style="margin-top:12px;display:flex;gap:10px;flex-wrap:wrap;">
        <button class="btn btn-red" type="submit" name="submit">Save Student</button>
        <a class="btn btn-white" href="student_list.php">Back to List</a>
      </div>
    </form>
  </div>
</div>

</body>
</html>